// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of the License "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// e32test\buffer\t_char.cpp
// Overview:
// Test TChar, TCharF, TChaLC and TCharUC character classes.
// API Information:
// TChar, TCharF, TChaLC, TCharUC.
// Details:
// - For different characters 
// - Verify character's standard category information
// - CR1080 verification of standard category information for additional Indic characters
// - Verify character's bi-directional Unicode character category.
// - Check if character is printable.
// - Check if character is a graphical character.
// - Verify for Chinese, Japanese, Korean (CJK) notional width.
// - Verify that it is upper case, lower case form.
// - Verify character's title case.
// - Verify mirrored property of character.
// - Verify integer numeric property of character.
// - Verify character can be decomposed.
// - For different unicode strings, verify unicode collation
// - Compare data of two descriptors using the different collation methods and 
// check results are as expected.
// - Call the methods in the TChar, TCharF, TCharUC and TCharUC
// - For TChar, TCharF, TChaLC, TCharUC
// - Check the constructor and operators are as expected.
// - Check folded character is same when it is converted to folded uppercase.
// - Check folded and uppercase character is same when it is converted to 
// uppercase letter stripped of accents.
// - Convert lowercase, uppercase characters to it's lowercase form, check the conversion 
// is as expected, and lowercase characters are unchanged.
// - Convert lowercase, uppercase characters to it's uppercase form, check the conversion 
// is as expected, and uppercase characters are unchanged.
// Platforms/Drives/Compatibility:
// All 
// Assumptions/Requirement/Pre-requisites:
// Failures and causes:
// Base Port information:
// 
//

#include <e32test.h>
#include <collate.h>	  

#ifdef __VC32__
    // Solve compilation problem caused by non-English locale
    #pragma setlocale("english")
#endif

LOCAL_D RTest test(_L("T_CHAR"));

enum
	{
	Ydieresis=0x178,
	nobreakspace=160,
	exclamdown,
	cent,
	sterling,
	currency,
	yen,
	brokenbar,
	section,
	dieresis,
	copyright,
	ordfeminine,
	leftdoubleanglequote,
	logicalnot,
	softhyphen,
	registered,
	macron,
	degree,
	plusminus,
	twosuperior,
	threesuperior,
	acute,
	micro,
	pilcrow,
	dotaccent,
	cedilla,
	onesuperior,
	ordmasculine,
	rightdoubleanglequote,
	onequarter,
	onehalf,
	threequarters,
	questiondown,
	Agrave,
	Aacute,
	Acircumflex,
	Atilde,
	Adieresis,
	Aring,
	AE,
	Ccedilla,
	Egrave,
	Eacute,
	Ecircumflex,
	Edieresis,
	Igrave,
	Iacute,
	Icircumflex,
	Idieresis,
	Eth,
	Ntilde,
	Ograve,
	Oacute,
	Ocircumflex,
	Otilde,
	Odieresis,
	multiplication,
	Oslash,
	Ugrave,
	Uacute,
	Ucircumflex,
	Udieresis,
	Yacute,
	Thorn,
	germanbls,
	agrave,
	aacute,
	acircumflex,
	atilde,
	adieresis,
	aring,
	ae,
	ccedilla,
	egrave,
	eacute,
	ecircumflex,
	edieresis,
	igrave,
	iacute,
	icircumflex,
	idieresis,
	eth,
	ntilde,
	ograve,
	oacute,
	ocircumflex,
	otilde,
	odieresis,
	division,
	oslash,
	ugrave,
	uacute,
	ucircumflex,
	udieresis,
	yacute,
	thorn,
	ydieresis,
	Dzcaron = 0x01C5, 
	Lj = 0x01C8, 
	Udearesismacron = 0x01D5,
	Adotabove = 0x01E0,
	AEmacron = 0x01E2,
	aringacute = 0x01FB,
	scommabelow = 0x0219, 
	Otildemacron = 0x022C,
	Cstroke = 0x023B,
	doubleprime = 0x02BA, 
	lowleftarrow = 0x02FF,
	candrabindu = 0x0310, 
	doublerightwardsarrowbelow = 0x0362,
	latinsmalla = 0x0363	
	};

struct TCharDump
	{
	TUint iChar;
	};

/*EXPORT_C*/ void TChar::__DbgTest(void* pTCharDump) const
	{
	((TCharDump*)pTCharDump)->iChar=iChar;
	}

typedef enum {eTChar, eTCharF, eTCharC, eTCharLC, eTCharUC} TCharTypes;

template<class TCharType>	
class TestTChar
	{
public:
	void Test1();
	void Test2(TCharTypes);
	void Test3(TCharTypes);
	void Test4(TCharTypes);
	};

template<class TCharType>
GLDEF_C void TestTChar<TCharType>::Test1()
// Test that the methods are in the DLL
	{
	test.Start(_L("Test methods are in DLL - TChar"));
	TChar a('X');
	TUint TU=0;
	TChar b(TU);
	TChar c(b);

	c=TU;
	c=a;
	c-=TU;
	c+=TU;
	c-TU;
	c+TU;
	c=(TUint)(a);

	a.Fold();
	a.LowerCase();
	a.UpperCase();
	a.Eos();
	a.IsLower();
	a.IsUpper();
	a.IsAlpha();
	a.IsDigit();
	a.IsAlphaDigit();
	a.IsHexDigit();
	a.IsSpace();
	a.IsPunctuation();
	a.IsGraph();
	a.IsPrint();
//	a.SetChar(TU); protected

	test.Next(_L("Test methods are in DLL - TCharF"));	
	TCharF d(TU);
	TCharF e(a);
    e=d;
	e=TU;
	e=a;

	test.Next(_L("Test methods are in DLL - TCharLC"));	
	TCharLC h(TU);
	TCharLC i(a);
    h=i;
	h=TU;
	h=a;

	test.Next(_L("Test methods are in DLL - TCharUC"));	
	TCharUC j(TU);
	TCharUC k(a);
    k=j;
	k=TU;
	k=a;
	test.End();
	}
			

//*********************************
// Test the constructors
//*********************************
template<class TCharType>
GLDEF_C void TestTChar<TCharType>::Test2(TCharTypes CharType)
	{
	test.Start(_L("Test constructors"));
	TCharDump dump;
	TCharType b('x');
	b.__DbgTest(&dump);
	switch(CharType)
		{
		case eTChar:	
		case eTCharLC:
			test(dump.iChar=='x');
			break;
		case eTCharUC:
			test(dump.iChar=='X');
			break;
		case eTCharF:
		case eTCharC:
			test(dump.iChar == 'x');
		}
	test.End();
	}


//*********************************
// Test the operators
//*********************************
template<class TCharType>
GLDEF_C void TestTChar<TCharType>::Test3(TCharTypes CharType)
	{
	test.Start(_L("Test the operators")); 
	test.Next(_L("="));
	TCharDump dump;
	TUint TUx='x', TUX='X';
	TChar TCy='y', TCY='Y';
	TCharType aChar('m');

	aChar=TUx;
	aChar.__DbgTest(&dump);
	switch(CharType)
		{
		case eTChar:	
		case eTCharLC:	test(dump.iChar=='x'); break;
		case eTCharUC:	test(dump.iChar=='X'); break;
		case eTCharF:
		case eTCharC:	
			test(dump.iChar == 'x');
		}

	aChar=TUX; 
	aChar.__DbgTest(&dump);
	switch(CharType)
		{	
		case eTCharLC:	test(dump.iChar=='x'); break;
		case eTChar:
		case eTCharUC:	test(dump.iChar=='X'); break;
		case eTCharF:
		case eTCharC:	
			test(dump.iChar == 'x');
		}

	aChar=TCy;
	aChar.__DbgTest(&dump);
	switch(CharType)
		{
		case eTChar:	
		case eTCharLC:	test(dump.iChar=='y'); break;
		case eTCharUC:	test(dump.iChar=='Y'); break;
		case eTCharF:
		case eTCharC:	
			test(dump.iChar == 'y');
		}

	aChar=TCY;
	aChar.__DbgTest(&dump);
	switch(CharType)
		{	
		case eTCharLC:	test(dump.iChar=='y'); break;
		case eTChar:
		case eTCharUC:	test(dump.iChar=='Y'); break;
		case eTCharF:
		case eTCharC:	
			test(dump.iChar == 'y');
		}


	test.Next(_L("-=, +=, -, + and cast"));
	TCharType x(10);
	x-=5;
	x.__DbgTest(&dump);
	test(dump.iChar==5);

	x+=5;
	x.__DbgTest(&dump);
	test(dump.iChar==10);

	test((x-(TUint)5)==5);
	x.__DbgTest(&dump);
	test(dump.iChar==10);

	test((x+(TUint)5)==15);
	x.__DbgTest(&dump);
	test(dump.iChar==10);

	TChar z('a');
	TUint w=z;
	test(w==97);

	test.End();
	}


//////////////////////////////////////////////
// Test Fold, Fold(TChar::EFoldAccents), Collate, LowerCase, UpperCase
////////////////////////////////////////////
template<class TCharType>
GLDEF_C void TestTChar<TCharType>::Test4(TCharTypes CharType)
	{
	test.Start(_L("Fold(), Fold(TChar::EFoldAccents), Collate(), LowerCase() UpperCase() & Eos()"));
	TCharDump dump1, dump2;
	TCharType c(0);

	// Is the folded version the same as the folded uppercase version?
	test.Next(_L("Fold() 0-127"));
    TUint i;
	for(i=0; i<=127; i++)
		{
		c = i;
		c.Fold();
		c.__DbgTest(&dump1);
		c.UpperCase();
		c.Fold();
		c.__DbgTest(&dump2);
		test(dump1.iChar == dump2.iChar);
		}

	// Is the folded and uppercased version the same as an uppercase letter stripped of accents?
	test.Next(_L("Fold() >127"));
	for(i=128; i<=255; i++)
		{
		c=i;
		c.Fold();
		c.UpperCase();
		c.__DbgTest(&dump1);
		if(CharType==eTChar)
			{
			switch(i)
				{
			case ccedilla:	
			case Ccedilla:
                test(dump1.iChar=='C');
                break;

			case udieresis:	
			case ucircumflex:
			case ugrave:
			case uacute:		
			case Udieresis:
			case Ucircumflex:
			case Ugrave:
			case Uacute:
                test(dump1.iChar=='U');
                break;

			case ydieresis:
			case yacute:
			case Yacute:
                test(dump1.iChar=='Y');
                break;

			case ecircumflex:
			case edieresis:
			case egrave:
			case eacute:		
			case Ecircumflex:
			case Edieresis:
			case Egrave:
			case Eacute:
                test(dump1.iChar=='E');
                break;

			case acircumflex:
			case adieresis:
			case agrave:		
			case aacute:
			case Acircumflex: 
			case Adieresis:	
			case Agrave:			
			case Aacute:
			case atilde:
			case Atilde:
			case aring:		
			case Aring:
                test(dump1.iChar=='A');
                break;

			case idieresis:
			case icircumflex:
			case iacute:
			case igrave:			
			case Idieresis:
			case Icircumflex:
			case Iacute:
			case Igrave:
                test(dump1.iChar=='I');
                break;

			case ocircumflex:
			case odieresis:
			case ograve:
			case oacute:		
			case otilde:
			case Ocircumflex:
			case Odieresis:
			case Ograve:
			case Oacute:
			case Otilde:
                test(dump1.iChar=='O');
                break;

			case ntilde:
			case Ntilde:
                test(dump1.iChar=='N'); break;

			case oslash:
			case Oslash:

				test(dump1.iChar==Oslash); break;

			case ae:
			case AE:
				test(dump1.iChar==AE); break;

			case eth:
			case Eth:
                test(dump1.iChar==Eth); break;

			case thorn:
			case Thorn:
                test(dump1.iChar==Thorn);
                break;

			case nobreakspace:
				test(dump1.iChar == ' ');
				break;

			case micro:
				test(dump1.iChar == 0x039C);
				break;

			default:
                test(dump1.iChar==i); 
				}
			}
		}
	
	// In the following test section We test Fold(TChar::EFoldAccents) - This code tests this function for all the
	// set of unicode characters located between 0x0080 and 0x00FF which are already tested for Fold(). 
	// I added to this test 13 unicode character located between 0x01C5 and 0x0363.
	
	test.Next(_L("Fold(TChar::EFoldAccents) >127"));
	for(i=128; i<=255; i++)
		{
		c=i;
		c.Fold(TChar::EFoldAccents);
		c.__DbgTest(&dump1);
		if(CharType==eTChar)
			{
			switch(i)
				{
			case ccedilla:
				test(dump1.iChar=='c');
				break;
				
			case Ccedilla:
				test(dump1.iChar=='C');
				break;
				
			case acircumflex:
			case adieresis:
			case agrave:		
			case aacute:
			case atilde:
			case aring:
			case aringacute:			
            	test(dump1.iChar=='a');
            	break;
            	
			case Acircumflex: 
			case Adieresis:	
			case Agrave:			
			case Aacute:
			case Atilde:		
			case Aring:		
                test(dump1.iChar=='A');
                break;

			case udieresis:	
			case ucircumflex:
			case ugrave:
			case uacute:
				test(dump1.iChar=='u');
				break;
				
			case Udieresis:
			case Ucircumflex:
			case Ugrave:
			case Uacute:
                test(dump1.iChar=='U');
                break;

			case ydieresis:
			case yacute:
				test(dump1.iChar=='y');
				break;
            
			case Yacute:
                test(dump1.iChar=='Y');
                break;

			case ecircumflex:
			case edieresis:
			case egrave:
			case eacute:		
            	test(dump1.iChar=='e');
            	break;
            	
			case Ecircumflex:
			case Edieresis:
			case Egrave:
			case Eacute:
                test(dump1.iChar=='E');
                break;			
				
			case idieresis:
			case icircumflex:
			case iacute:
			case igrave:
				test(dump1.iChar=='i');
				break;
				
			case Idieresis:
			case Icircumflex:
			case Iacute:
			case Igrave:
                test(dump1.iChar=='I');
                break;

			case ocircumflex:
			case odieresis:
			case ograve:
			case oacute:		
			case otilde:
				test(dump1.iChar=='o');
				break;
			
			case Ocircumflex:
			case Odieresis:
			case Ograve:
			case Oacute:
			case Otilde:
                test(dump1.iChar=='O');
                break;

			case ntilde: test(dump1.iChar=='n'); break;
				
			case Ntilde: test(dump1.iChar=='N'); break;

			case oslash: test(dump1.iChar==oslash); break;
			case Oslash: test(dump1.iChar==Oslash); break;

			case ae: 	test(dump1.iChar==ae); break;
			case AE: 	test(dump1.iChar==AE); break;

			case eth:   test(dump1.iChar==eth); break;
			case Eth:   test(dump1.iChar==Eth); break;

			case thorn: test(dump1.iChar==thorn);break;
			case Thorn: test(dump1.iChar==Thorn);break;

			case micro:
				test(dump1.iChar == 0x00B5);
				break;
							
			default:
                test(dump1.iChar==i); 
				}
			}
		}
	
	TUint specialCasesTable [] = 
		{
		0x01C5, // Dzcaron
		0x01C8, // Lj
		0x01D5,//  Udearesismacron
		0x01E0,//  Adotabove
		0x01E2,//  AEmacron
		0x0219, // scommabelow
		0x022C,//  Otildemacron
		0x023B,//  Cstroke
		0x02BA, // doubleprime
		0x02FF,//  lowleftarrow
		0x0310, // candrabindu
		0x0362, // doublerightwardsarrowbelow
		0x0363 //  latinsmalla
		};
	    
	for (i=0;i<=12; i++)
		{
		c=specialCasesTable[i];
		c.Fold(TChar::EFoldAccents);
		c.__DbgTest(&dump1);
		if(CharType==eTChar)
			{
			switch(specialCasesTable[i])
				{
			case Dzcaron					:	test(dump1.iChar==Dzcaron);	break; 
			case Lj							:	test(dump1.iChar==Lj);		break; 
			case Udearesismacron			:	test(dump1.iChar=='U');		break; 
			case Adotabove					:	test(dump1.iChar=='A');		break;
			case AEmacron					:	test(dump1.iChar==AE);		break; 
			case scommabelow				:	test(dump1.iChar=='s');		break;
			case Otildemacron				:	test(dump1.iChar=='O');		break; 
			case Cstroke					:	test(dump1.iChar==Cstroke);			break; 
			case doubleprime				:	test(dump1.iChar==doubleprime);		break; 
			case lowleftarrow				:	test(dump1.iChar==lowleftarrow);	break; 
			case candrabindu				:	test(dump1.iChar==candrabindu);		break; 
			case doublerightwardsarrowbelow	:	test(dump1.iChar==doublerightwardsarrowbelow);	break; 
			case latinsmalla 				:	test(dump1.iChar==latinsmalla);		break; 
			default							:	test(dump1.iChar==specialCasesTable[i]); 
				}
			}
		}

	test.Next(_L("LowerCase() 0-127"));
	for(i=0;i<=127; i++)
		{ 
		c=i;
		if(c.IsUpper())
			{
			c.__DbgTest(&dump1);
			c.LowerCase();
			c.__DbgTest(&dump2);
			test(dump1.iChar==dump2.iChar-32);
			}
		else
			{
			c.__DbgTest(&dump1);
			c.LowerCase();
			c.__DbgTest(&dump2);
			test(dump1.iChar==dump2.iChar);
			}
		}


	test.Next(_L("LowerCase() >127"));
	for(i=128; i<=255; i++)
		{
		c=i;
		c.LowerCase();
		c.__DbgTest(&dump1);
		if(CharType==eTChar)
			{
			switch(i)
				{
			case Acircumflex:	test(dump1.iChar==acircumflex);	break;
			case Adieresis:		test(dump1.iChar==adieresis);	break;
			case Agrave:		test(dump1.iChar==agrave);		break;
			case Aacute:		test(dump1.iChar==aacute);		break;
			case Atilde:		test(dump1.iChar==atilde);		break;
			case Aring:			test(dump1.iChar==aring);		break;
			case AE:			test(dump1.iChar==ae);			break;
			case Ccedilla:		test(dump1.iChar==ccedilla); 	break;
			case Ecircumflex:	test(dump1.iChar==ecircumflex);	break;
			case Edieresis:		test(dump1.iChar==edieresis);	break;
			case Egrave:		test(dump1.iChar==egrave);		break;
			case Eacute:		test(dump1.iChar==eacute);		break;
			case Eth:			test(dump1.iChar==eth);			break;
			case Icircumflex:	test(dump1.iChar==icircumflex);	break;
			case Idieresis:		test(dump1.iChar==idieresis);	break;
			case Igrave:		test(dump1.iChar==igrave);		break;
			case Iacute:		test(dump1.iChar==iacute);		break;
			case Ntilde:		test(dump1.iChar==ntilde);		break;
			case Oslash:		test(dump1.iChar==oslash);		break;
			case Ocircumflex:	test(dump1.iChar==ocircumflex);	break;
			case Odieresis:		test(dump1.iChar==odieresis);	break;
			case Ograve:		test(dump1.iChar==ograve);		break;
			case Oacute:		test(dump1.iChar==oacute);		break;
			case Otilde:		test(dump1.iChar==otilde);		break;
			case Thorn:			test(dump1.iChar==thorn);		break;
			case Ucircumflex:	test(dump1.iChar==ucircumflex);	break;
			case Udieresis:		test(dump1.iChar==udieresis);	break;
			case Ugrave:		test(dump1.iChar==ugrave);		break;
			case Uacute:		test(dump1.iChar==uacute);		break;
			case Yacute:		test(dump1.iChar==yacute);		break;
			default: 			test(dump1.iChar==i); 		
				}
			}
		}


	test.Next(_L("UpperCase() 0-127"));
	for(i=0;i<=127; i++)
		{ 
		c=i;
		if(c.IsLower())
			{
			c.__DbgTest(&dump1);
			c.UpperCase();
			c.__DbgTest(&dump2);
			test(dump1.iChar==dump2.iChar+32);
			}
		else
			{
			c.__DbgTest(&dump1);
			c.UpperCase();
			c.__DbgTest(&dump2);
			test(dump1.iChar==dump2.iChar);
			}
		}


	test.Next(_L("UpperCase() >127"));
	for(i=128; i<=255; i++)
		{
		c=i;
		c.UpperCase();
		c.__DbgTest(&dump1);
		if(CharType==eTChar)
			{
			switch(i)	
				{
			case acircumflex:	test(dump1.iChar==Acircumflex);	break;
			case adieresis:		test(dump1.iChar==Adieresis);	break;
			case agrave:		test(dump1.iChar==Agrave);		break;
			case aacute:		test(dump1.iChar==Aacute);		break;
			case atilde:		test(dump1.iChar==Atilde);		break;
			case aring:			test(dump1.iChar==Aring);		break;
			case ae:			test(dump1.iChar==AE);			break;
			case ccedilla:		test(dump1.iChar==Ccedilla); 	break;
			case ecircumflex:	test(dump1.iChar==Ecircumflex);	break;
			case edieresis:		test(dump1.iChar==Edieresis);	break;
			case egrave:		test(dump1.iChar==Egrave);		break;
			case eacute:		test(dump1.iChar==Eacute);		break;
			case eth:			test(dump1.iChar==Eth);			break;
			case icircumflex:	test(dump1.iChar==Icircumflex);	break;
			case idieresis:		test(dump1.iChar==Idieresis);	break;
			case igrave:		test(dump1.iChar==Igrave);		break;
			case iacute:		test(dump1.iChar==Iacute);		break;
			case ntilde:		test(dump1.iChar==Ntilde);		break;
			case oslash:		test(dump1.iChar==Oslash);		break;
			case ocircumflex:	test(dump1.iChar==Ocircumflex);	break;
			case odieresis:		test(dump1.iChar==Odieresis);	break;
			case ograve:		test(dump1.iChar==Ograve);		break;
			case oacute:		test(dump1.iChar==Oacute);		break;
			case otilde:		test(dump1.iChar==Otilde);		break;
			case thorn:			test(dump1.iChar==Thorn);		break;
			case ucircumflex:	test(dump1.iChar==Ucircumflex);	break;
			case udieresis:		test(dump1.iChar==Udieresis);	break;
			case ugrave:		test(dump1.iChar==Ugrave);		break;
			case uacute:		test(dump1.iChar==Uacute);		break;
			case yacute:		test(dump1.iChar==Yacute);		break;
			case ydieresis:		test(dump1.iChar==Ydieresis);	break;
			case micro:			test(dump1.iChar==0x039C);		break;
			default: 			test(dump1.iChar==i); 		
				}
			}
		}
		   
	test.Next(_L("Eos()"));
	c=0;
	test(c.Eos()==TRUE);
	for(i=1;i<=255;i++)
		{
		c=i;
		test(c.Eos()==FALSE);
		}
	test.End();	
	}


//*********************************
// Test the Isxxx functions
//*********************************
//
// Deleted as it can no longer use the C library
//
/*
template<class TCharType>
GLDEF_C void TestTChar<TCharType>::Test5(TCharTypes CharType)
	{
	test.Start(_L("Isxxxxx()"));
	test.End();
	}
*/

#ifndef _DEBUG
#pragma warning (disable :4710) //Function not expanded 
#pragma warning (disable :4700) //Local variable used without being initialized
#endif

/*
Put all the information about a character into a string. This can then be compared with
a previously verified string to test the character classification functions.

To keep the string short, only non-default information is appended, except for the first
attribute, the category.

The format is:

1. The two-letter Unicode category code: e.g., Lu for an uppercase letter, Nd for a digit, etc.
2. If the bidirectional category is not TChar::ELeftToRight, ' b=' then the one or two-letter code.
3. If the combining class is not 0, ' c=' then the combining class number in decimal.
4. If the uppercase form is different from the character code, ' u=' then the uppercase form in hex.
5. If the titlecase form is different from the character code, ' t=' then the titlecase form in hex.
6. If the lowercase form is different from the character code, ' l=' then the lowercase form in hex.
** REMOVED ** 7. If the folded form is different from the character code, ' f=' then the folded form in hex.
8. If the character has the mirrored property, ' m'.
9. If the character has a numeric value, ' n=' then the numeric value in decimal, or -2 for fractions.
10. If the CJK width is not neutral, ' w=' then 1 for halfwidth, 2 for fullwidth, 3 for narrow, 4 for wide
11. If there is a canonical decomposition, ' d=' then the characters of the decomposition in hex, comma-separated.
*/
void char_info_to_string(TDes16& info,TUint x)
	{
	TChar c(x);

	TChar::TCharInfo ci;
	c.GetInfo(ci);

	TBuf<16> cat;
	switch (ci.iCategory)
		{
		case TChar::ELuCategory: cat = _L("Lu"); break;
		case TChar::ELlCategory: cat = _L("Ll"); break;
		case TChar::ELtCategory: cat = _L("Lt"); break; 
		case TChar::ELoCategory: cat = _L("Lo"); break; 
		case TChar::ELmCategory: cat = _L("Lm"); break; 
		case TChar::EMnCategory: cat = _L("Mn"); break; 
		case TChar::EMcCategory: cat = _L("Mc"); break; 
		case TChar::ENdCategory: cat = _L("Nd"); break; 
		case TChar::ENlCategory: cat = _L("Nl"); break;
		case TChar::ENoCategory: cat = _L("No"); break; 
		case TChar::EPdCategory: cat = _L("Pd"); break; 
		case TChar::EPsCategory: cat = _L("Ps"); break; 
		case TChar::EPeCategory: cat = _L("Pe"); break; 
		case TChar::EPoCategory: cat = _L("Po"); break; 
		case TChar::EPiCategory: cat = _L("Pi"); break; 
		case TChar::EPfCategory: cat = _L("Pf"); break; 
		case TChar::ESmCategory: cat = _L("Sm"); break; 
		case TChar::EScCategory: cat = _L("Sc"); break; 
		case TChar::ESkCategory: cat = _L("Sk"); break;
		case TChar::ESoCategory: cat = _L("So"); break; 
		case TChar::EZsCategory: cat = _L("Zs"); break; 
		case TChar::EZlCategory: cat = _L("Zl"); break; 
		case TChar::EZpCategory: cat = _L("Zp"); break; 
		case TChar::ECcCategory: cat = _L("Cc"); break; 
		case TChar::ECoCategory: cat = _L("Co"); break; 
		case TChar::ECnCategory: cat = _L("Cn"); break;
		case TChar::ECfCategory: cat = _L("Cf"); break;
		default: User::Panic(_L("bad char category"),0); break; 
		}
	info.Append(cat);

	TBuf<16> bdcat;
	switch (ci.iBdCategory)
		{
		case TChar::ELeftToRight: bdcat=_L("L"); break;
		case TChar::ELeftToRightEmbedding: bdcat=_L("LRE"); break;
		case TChar::ELeftToRightOverride: bdcat=_L("LRO"); break;
		case TChar::ERightToLeft: bdcat=_L("R"); break;
		case TChar::ERightToLeftArabic: bdcat=_L("AL"); break; 
		case TChar::ERightToLeftEmbedding: bdcat=_L("RLE"); break;
		case TChar::ERightToLeftOverride: bdcat=_L("RLO"); break;
		case TChar::EPopDirectionalFormat: bdcat=_L("PDF"); break;
		case TChar::EEuropeanNumber: bdcat=_L("EN"); break; 
		case TChar::EEuropeanNumberSeparator: bdcat=_L("ES"); break;  
		case TChar::EEuropeanNumberTerminator: bdcat=_L("ET"); break;
		case TChar::EArabicNumber: bdcat=_L("AN"); break; 
		case TChar::ECommonNumberSeparator: bdcat=_L("CS"); break; 
		case TChar::ENonSpacingMark: bdcat=_L("NSM"); break;
		case TChar::EBoundaryNeutral: bdcat=_L("BN"); break; 
		case TChar::EParagraphSeparator: bdcat=_L("B"); break;
		case TChar::ESegmentSeparator: bdcat=_L("S"); break;
		case TChar::EWhitespace: bdcat=_L("WS"); break; 
		case TChar::EOtherNeutral: bdcat=_L("ON"); break; 
		default: User::Panic(_L("bad bd category"),0); break; 
		}
	if (ci.iBdCategory != TChar::ELeftToRight)
		{
		info.Append(_L(" b="));
		info.Append(bdcat);
		}

	if (ci.iCombiningClass != 0)
		{
		info.Append(_L(" c="));
		info.AppendNum(ci.iCombiningClass);
		}

	if (ci.iUpperCase != x)
		{
		info.Append(_L(" u="));
		info.AppendNum(ci.iUpperCase,EHex);
		}
		
	if (ci.iTitleCase != x)
		{
		info.Append(_L(" t="));
		info.AppendNum(ci.iTitleCase,EHex);
		}

	if (ci.iLowerCase != x)
		{
		info.Append(_L(" l="));
		info.AppendNum(ci.iLowerCase,EHex);
		}

	if (ci.iMirrored)
		info.Append(_L(" m"));

	if (ci.iNumericValue != -1)
		{
		info.Append(_L(" n="));
		info.AppendNum(ci.iNumericValue);
		}

	switch (c.GetCjkWidth())
		{
		case TChar::EHalfWidth: info.Append(_L(" w=1")); break;
		case TChar::EFullWidth: info.Append(_L(" w=2")); break;
		case TChar::ENarrow: info.Append(_L(" w=3")); break;
		case TChar::EWide: info.Append(_L(" w=4")); break;
		default: break;
		}

	TPtrC16 d;
	if (c.Decompose(d))
		{
		info.Append(_L(" d="));
		for (int i = 0; i < d.Length(); i++)
			{
			info.AppendNum((TUint)d[i],EHex);
			if (i < d.Length() - 1)
				info.Append(',');
			}
		}

	if (!c.IsPrint())
		info.Append(_L(" np"));
	if (!c.IsGraph())
		info.Append(_L(" ng"));
	}

void test_unicode_char(TUint c,const TDesC& desired_info,const TDesC& name)
	{
	test.Next(name);
	TBuf<128> info;
	char_info_to_string(info,c);
//	test.Printf(_L("code=0x%x desired='%S' actual='%S'\n"),c,&desired_info,&info);
//	if (info != desired_info)
//		test.Printf(_L(" (mismatch!)\n"));
	test(info == desired_info);
	}

void test_unicode_chars()
	{
	test_unicode_char(0x0000,_L("Cc b=BN np ng"),_L("NULL"));
	test_unicode_char(0x000A,_L("Cc b=B np ng"),_L("linefeed"));
	test_unicode_char(0x0020,_L("Zs b=WS w=3 ng"),_L("space"));
	test_unicode_char(0x0028,_L("Ps b=ON m w=3"),_L("("));
	test_unicode_char(0x0038,_L("Nd b=EN n=8 w=3"),_L("8"));
	test_unicode_char(0x0040,_L("Po b=ON w=3"),_L("@"));
	test_unicode_char(0x0062,_L("Ll u=42 w=3"),_L("b"));
	test_unicode_char(0x0042,_L("Lu l=62 w=3"),_L("B"));
	test_unicode_char(0x0067,_L("Ll u=47 w=3"),_L("g"));
	test_unicode_char(0x0047,_L("Lu l=67 w=3"),_L("G"));
	test_unicode_char(0x0077,_L("Ll u=57 w=3"),_L("w"));
	test_unicode_char(0x0057,_L("Lu l=77 w=3"),_L("W"));
	test_unicode_char(0x00A0,_L("Zs b=CS ng"),_L("non-break space"));
	test_unicode_char(0x00BC,_L("No b=ON n=-2"),_L("1/4"));
	test_unicode_char(0x00DD,_L("Lu l=fd d=59,301"),_L("Y acute"));
	test_unicode_char(0x00FF,_L("Ll u=178 d=79,308"),_L("y umlaut"));
	test_unicode_char(0x0178,_L("Lu l=ff d=59,308"),_L("Y umlaut"));
	test_unicode_char(0x01C4,_L("Lu t=1c5 l=1c6"),_L("DZ caron"));
	test_unicode_char(0x01C5,_L("Lt u=1c4 l=1c6"),_L("Dz caron"));
	test_unicode_char(0x01C6,_L("Ll u=1c4 t=1c5"),_L("dz caron"));
	test_unicode_char(0x022A,_L("Lu l=22b d=4f,308,304"),_L("O with diaeresis and macron"));
	test_unicode_char(0x0267,_L("Ll"),_L("latin small letter heng with hook"));
	test_unicode_char(0x02b0,_L("Lm"),_L("modifier letter small h"));
	test_unicode_char(0x030a,_L("Mn b=NSM c=230"),_L("combining ring above"));
	test_unicode_char(0x03a8,_L("Lu l=3c8"),_L("Greek Psi"));
	test_unicode_char(0x03c8,_L("Ll u=3a8"),_L("Greek psi"));
	test_unicode_char(0x0401,_L("Lu l=451 d=415,308"),_L("Cyrillic IO"));
	test_unicode_char(0x0451,_L("Ll u=401 d=435,308"),_L("Cyrillic io"));
	test_unicode_char(0x04e0,_L("Lu l=4e1"),_L("Cyrillic Abkhasian Dze"));
	test_unicode_char(0x04e1,_L("Ll u=4e0"),_L("Cyrillic Abkhasian dze"));
	test_unicode_char(0x05A8,_L("Mn b=NSM c=230"),_L("Hebrew accent qadma"));
	test_unicode_char(0x05D2,_L("Lo b=R"),_L("Hebrew letter gimel"));
	test_unicode_char(0x05F0,_L("Lo b=R"),_L("Hebrew ligature Yiddish double vav"));
	test_unicode_char(0x0627,_L("Lo b=AL"),_L("Arabic letter alef"));
	test_unicode_char(0x0660,_L("Nd b=AN n=0"),_L("Arabic-Indic digit 0"));
	test_unicode_char(0x0669,_L("Nd b=AN n=9"),_L("Arabic-Indic digit 9"));
	test_unicode_char(0x06F4,_L("Nd b=EN n=4"),_L("Extended Arabic-Indic digit 4"));
	test_unicode_char(0x0B9C,_L("Lo"),_L("Tamil letter ja"));
	test_unicode_char(0x0BCD,_L("Mn b=NSM c=9"),_L("Tamil sign virama"));
	test_unicode_char(0x0BF0,_L("No n=10"),_L("Tamil number 10"));
	test_unicode_char(0x0BF1,_L("No n=100"),_L("Tamil number 100"));
	test_unicode_char(0x0BF2,_L("No n=1000"),_L("Tamil number 1000"));
	test_unicode_char(0x0E3F,_L("Sc b=ET"),_L("Thai currency symbol baht"));
	test_unicode_char(0x0ED5,_L("Nd n=5"),_L("Lao digit 5"));
	test_unicode_char(0x104E,_L("Po"),_L("Myanmar symbol aforementioned"));
	test_unicode_char(0x124B,_L("Lo"),_L("Ethiopic syllable QWAA"));
	test_unicode_char(0x137C,_L("No n=10000"),_L("Ethiopic number 10000"));
	test_unicode_char(0x217F,_L("Nl u=216f n=1000"),_L("small Roman numeral 1000"));
	test_unicode_char(0x2180,_L("Nl n=1000"),_L("Roman numeral 1000 CD"));
	test_unicode_char(0x2181,_L("Nl n=5000"),_L("Roman numeral 5000"));
	test_unicode_char(0x2182,_L("Nl n=10000"),_L("Roman numeral 10000"));
	test_unicode_char(0x2208,_L("Sm b=ON m"),_L("element of"));
	test_unicode_char(0x2248,_L("Sm b=ON m"),_L("almost equal to"));
	test_unicode_char(0x3000,_L("Zs b=WS w=4 ng"),_L("ideographic space"));
	test_unicode_char(0x3030,_L("Pd b=ON w=4"),_L("wavy dash"));
	test_unicode_char(0x305F,_L("Lo w=4"),_L("Hiragana ta"));
	test_unicode_char(0x3060,_L("Lo w=4 d=305f,3099"),_L("Hiragana da"));
	test_unicode_char(0x3099,_L("Mn b=NSM c=8 w=4"),_L("combining Katakana-Hiragana voiced sound mark"));
	test_unicode_char(0x30B3,_L("Lo w=4"),_L("Katakana ko"));
	test_unicode_char(0x30B4,_L("Lo w=4 d=30b3,3099"),_L("Katakana go"));
	test_unicode_char(0x51FF,_L("Lo w=4"),_L("CJK ideograph 51FF"));
	test_unicode_char(0x52FF,_L("Lo w=4"),_L("CJK ideograph 52FF"));
	test_unicode_char(0xCEFF,_L("Lo w=4"),_L("Hangul syllable CEFF"));
	test_unicode_char(0xFF17,_L("Nd b=EN n=7 w=2"),_L("fullwidth digit 7"));
	test_unicode_char(0xFF33,_L("Lu l=ff53 w=2"),_L("fullwidth S"));
	test_unicode_char(0xFF53,_L("Ll u=ff33 w=2"),_L("fullwidth s"));
	test_unicode_char(0xFF85,_L("Lo w=1"),_L("halfwidth Katakana na"));
	test_unicode_char(0xFFE1,_L("Sc b=ET w=2"),_L("fullwidth pound sign"));
	test_unicode_char(0xE000, _L("Co"), _L("End user private use character"));
	test_unicode_char(0xE3FF, _L("Co"), _L("End user private use character"));
	test_unicode_char(0xE400, _L("Co"), _L("Corporate private use character"));
	test_unicode_char(0xF6DA, _L("Co np ng"), _L("Corporate private use character - Java string marker"));
	test_unicode_char(0xF6DB, _L("Co"), _L("Corporate private use character - Replacement character"));
	test_unicode_char(0xF6DC, _L("Co np ng"), _L("Corporate private use character - switch to next"));
	test_unicode_char(0xF6DD, _L("Co np ng"), _L("Corporate private use character - switch to previous"));
	test_unicode_char(0xF6DB, _L("Co"), _L("Corporate private use character - Latin superscript y"));
	test_unicode_char(0xF6FF, _L("Co"), _L("Corporate private use character - Alarm bell"));
	test_unicode_char(0xF700, _L("Co np ng"), _L("Corporate private use character - Key code"));
	test_unicode_char(0xF8FF, _L("Co np ng"), _L("Corporate private use character - Key code"));
	
	// CR1080 - Testing additional Indic Chars
	// Test properties of new Indic characters added in CR1080, and also the properties of
	// existing characters immediately preceding and proceding the new characters, to test
	// for any regression.
	test_unicode_char(0x0903, _L("Mc"), _L("DEVANAGARI SIGN VISARGA"));  		// Existing
	test_unicode_char(0x0904, _L("Lo"), _L("DEVANAGARI LETTER SHORT A"));		// New
	test_unicode_char(0x0905, _L("Lo"), _L("DEVANAGARI LETTER A"));				// Existing
	
	test_unicode_char(0x0970, _L("Po"), _L("DEVANAGARI ABBREVIATION SIGN"));	// Existing
	test_unicode_char(0x097B, _L("Lo"), _L("DEVANAGARI LETTER GGA"));			// New
	test_unicode_char(0x097C, _L("Lo"), _L("DEVANAGARI LETTER JJA"));			// New
	test_unicode_char(0x097D, _L("Lo"), _L("DEVANAGARI LETTER GLOTTAL STOP"));	// New
	test_unicode_char(0x097E, _L("Lo"), _L("DEVANAGARI LETTER DDDA"));			// New
	test_unicode_char(0x097F, _L("Lo"), _L("DEVANAGARI LETTER BBA"));			// New
	test_unicode_char(0x0981, _L("Mn b=NSM"), _L("BENGALI SIGN CANDRABINDU"));	// Existing
	
	test_unicode_char(0x09BC, _L("Mn b=NSM c=7"), _L("BENGALI SIGN NUKTA"));	// Existing
	test_unicode_char(0x09BD, _L("Lo"), _L("BENGALI SIGN AVAGRAHA"));			// New
	test_unicode_char(0x09BE, _L("Mc"), _L("BENGALI VOWEL SIGN AA"));			// Existing
	
	test_unicode_char(0x09CD, _L("Mn b=NSM c=9"), _L("BENGALI SIGN VIRAMA"));	// Existing
	test_unicode_char(0x09CE, _L("Lo"), _L("BENGALI LETTER KHANDA TA"));		// New
	test_unicode_char(0x09D7, _L("Mc"), _L("BENGALI AU LENGTH MARK"));			// Existing

	test_unicode_char(0x09FA, _L("So"), _L("BENGALI ISSHAR"));					// Existing
	test_unicode_char(0x0A01, _L("Mn b=NSM"), _L("GURMUKHI SIGN ADAK BINDI"));	// New
	test_unicode_char(0x0A02, _L("Mn b=NSM"), _L("GURMUKHI SIGN BINDI"));		// Existing
	test_unicode_char(0x0A03, _L("Mc"), _L("GURMUKHI SIGN VISARGA"));			// New
	test_unicode_char(0x0A05, _L("Lo"), _L("GURMUKHI LETTER A"));				// Existing
	
	test_unicode_char(0x0A8B, _L("Lo"), _L("GUJARATI LETTER VOCALIC R"));		// Existing
	test_unicode_char(0x0A8C, _L("Lo"), _L("GUJARATI LETTER VOCALIC L"));		// New
	test_unicode_char(0x0A8D, _L("Lo"), _L("GUJARATI VOWEL CANDRA E"));			// Existing
	
	test_unicode_char(0x0AE0, _L("Lo"), _L("GUJARATI LETTER VOCALIC RR"));				// Existing
	test_unicode_char(0x0AE1, _L("Lo"), _L("GUJARATI LETTER VOCALIC LL"));				// New
	test_unicode_char(0x0AE2, _L("Mn b=NSM"), _L("GUJARATI VOWEL SIGN VOCALIC L"));		// New
	test_unicode_char(0x0AE3, _L("Mn b=NSM"), _L("GUJARATI VOWEL SIGN VOCALIC LL"));	// New
	test_unicode_char(0x0AE6, _L("Nd n=0"), _L("GUJARATI DIGIT ZERO")); 				// Existing
	
	test_unicode_char(0x0AEF, _L("Nd n=9"), _L("GUJARATI DIGIT NINE"));			// Existing
	test_unicode_char(0x0AF1, _L("Sc b=ET"), _L("GUJARATI RUPEE SIGN"));		// New
	test_unicode_char(0x0B01, _L("Mn b=NSM"), _L("ORIYA SIGN CANDRABINDU")); 	// Existing

	test_unicode_char(0x0B33, _L("Lo"), _L("ORIYA LETTER LLA"));				// Existing
	test_unicode_char(0x0B35, _L("Lo"), _L("ORIYA LETTER VA"));					// New
	test_unicode_char(0x0B36, _L("Lo"), _L("ORIYA LETTER SHA"));				// Existing

	test_unicode_char(0x0B70, _L("So"), _L("ORIYA ISSHAR"));					// Existing
	test_unicode_char(0x0B71, _L("Lo"), _L("ORIYA LETTER WA"));					// New
	test_unicode_char(0x0B82, _L("Mn b=NSM"), _L("TAMIL SIGN ANUSVARA")); 		// Existing

	test_unicode_char(0x0BB5, _L("Lo"), _L("TAMIL LETTER VA"));					// Existing
	test_unicode_char(0x0BB6, _L("Lo"), _L("TAMIL LETTER SHA"));				// New
	test_unicode_char(0x0BB7, _L("Lo"), _L("TAMIL LETTER SSA"));				// Existing

	test_unicode_char(0x0BD7, _L("Mc"), _L("TAMIL AU LENGTH MARK"));			// Existing
	test_unicode_char(0x0BE6, _L("Nd n=0"), _L("TAMIL DIGIT ZERO"));			// New
	test_unicode_char(0x0BE7, _L("Nd n=1"), _L("TAMIL DIGIT ZERO"));			// Existing

	test_unicode_char(0x0BF2, _L("No n=1000"), _L("TAMIL NUMBER ONE THOUSAND"));// Existing
	test_unicode_char(0x0BF3, _L("So b=ON"), _L("TAMIL DAY SIGN"));				// New
	test_unicode_char(0x0BF4, _L("So b=ON"), _L("TAMIL MONTH SIGN"));			// New
	test_unicode_char(0x0BF5, _L("So b=ON"), _L("TAMIL YEAR SIGN"));			// New
	test_unicode_char(0x0BF6, _L("So b=ON"), _L("TAMIL DEBIT SIGN"));			// New
	test_unicode_char(0x0BF7, _L("So b=ON"), _L("TAMIL CREDIT SIGN"));			// New
	test_unicode_char(0x0BF8, _L("So b=ON"), _L("TAMIL AS ABOVE SIGN"));		// New
	test_unicode_char(0x0BF9, _L("Sc b=ET"), _L("TAMIL RUPEE SIGN"));			// New
	test_unicode_char(0x0BFA, _L("So b=ON"), _L("TAMIL NUMBER SIGN"));			// New
	test_unicode_char(0x0C01, _L("Mc"), _L("TELUGU SIGN CANDRABINDU"));			// Existing
	
	test_unicode_char(0x0CB9, _L("Lo"), _L("KANNADA LETTER HA"));				// Existing
	test_unicode_char(0x0CBC, _L("Mn b=NSM c=7"), _L("KANNADA SIGN NUKTA"));	// New
	test_unicode_char(0x0CBD, _L("Lo"), _L("KANNADA SIGN AVAGRAHA"));			// New
	test_unicode_char(0x0CBE, _L("Mc"), _L("KANNADA VOWEL SIGN AA"));			// Existing

	test_unicode_char(0x0CE1, _L("Lo"), _L("KANNADA LETTER VOCALIC LL"));			// Existing
	test_unicode_char(0x0CE2, _L("Mn b=NSM"), _L("KANNADA VOWEL SIGN VOCALIC L"));	// New
	test_unicode_char(0x0CE3, _L("Mn b=NSM"), _L("KANNADA VOWEL SIGN VOCALIC LL"));	// New
	test_unicode_char(0x0CE6, _L("Nd n=0"), _L("KANNADA DIGIT ZERO"));				// Existing

	test_unicode_char(0x0CEF, _L("Nd n=9"), _L("KANNADA DIGIT NINE"));				// Existing
	test_unicode_char(0x0CF1, _L("So b=ON"), _L("KANNADA SIGN JIHVAMULIYA"));		// New
	test_unicode_char(0x0CF2, _L("So b=ON"), _L("KANNADA SIGN UPADHMANIYA"));		// New
	test_unicode_char(0x0D02, _L("Mc"), _L("MALAYALAM SIGN ANUSVARA"));				// Existing
	
	test_unicode_char(0x0B83, _L("Lo"), _L("TAMIL SIGN VISARGA"));					// Modified	
	test_unicode_char(0x0CBF, _L("Mn"), _L("KANNADA VOWEL SIGN I"));				// Modified
	test_unicode_char(0x0CC6, _L("Mn"), _L("KANNADA VOWEL SIGN E"));				// Modified

	// Unicode 5.0 specific
	test_unicode_char(0x0220, _L("Lu l=19e"), _L("LATIN CAPITAL LETTER N WITH LONG RIGHT LEG"));		// New
	test_unicode_char(0x0294, _L("Lo"), _L("LATIN LETTER GLOTTAL STOP"));								// Modified
	test_unicode_char(0x10400, _L("Lu l=10428"), _L("DESERET CAPITAL LETTER LONG I"));				// out BMP fold
	// Don't test those title case equal to upper case. See TUnicode::GetTitleCase().
	//test_unicode_char(0x10428, _L("Ll u=10400 t=10400"), _L("DESERET SMALL LETTER LONG I"));				// out BMP fold
	test_unicode_char(0x10428, _L("Ll u=10400"), _L("DESERET SMALL LETTER LONG I"));				// out BMP fold
	test_unicode_char(0x2F85B, _L("Lo w=4 d=58f7"), _L("CJK COMPATIBILITY IDEOGRAPH-2F85B"));			// New
	}

void test_unicode_collation(const TDesC& x,const TDesC& y,TInt desired_order,
							TInt level = 3,const TCollationMethod* method = NULL)
	{
	TInt order = x.CompareC(y,level,method);

	// Normalise order to -1, 0 or 1.
	if (order < 0)
		order = -1;
	else if (order > 0)
		order = 1;

	test(order == desired_order);
	}

static const TUint32 TheSwedishKey[] = 
	{
	0x8f60109,0x8f70109,0x8f80109,0x8f60121,0x8f70121,0x8f80121,0x8dd0109,0x8dd0121,
	0x8c50121,0x8c50109,
	};

static const TUint32 TheSwedishIndex[] = 
	{
	0x570008,0x770009,
	};

static const TUint16 TheSwedishStringElement[] = 
	{
	0x2,0x61,0x30a,0x2,0x61,0x308,0x2,0x6f,
	0x308,0x2,0x41,0x30a,0x2,0x41,0x308,0x2,
	0x4f,0x308,0x2,0x75,0x308,0x2,0x55,0x308,
	};

static const TUint32 TheSwedishStringIndex[] = 
	{
	0xc0004,0x90003,0xf0005,0x150007,0x30001,0x0,0x60002,0x120006,
	};

static const TCollationKeyTable TheSwedishTable = 
	{ TheSwedishKey, TheSwedishIndex, 2, TheSwedishStringElement, TheSwedishStringIndex, 8 };

static const TCollationMethod TheSwedishMethod =
	{ 0, NULL, &TheSwedishTable, 0 };

static const TCollationMethod TheSwapCaseMethod =
	{ 0, NULL, NULL, TCollationMethod::ESwapCase };

static const TCollationMethod TheFoldCaseMethod =
	{ 0, NULL, NULL, TCollationMethod::EFoldCase };

static const TUint32 TheKanaKey[] = 
	{
	0xc949,0xca09,0x2010f,0x20113,0x4010b,0x4010f,0x5010b,0x5010f,
	0x8010b,0x8010f,0xa010f,0xa014b,0xc010b,0xc010f,0xe010b,0xe010f,
	0xe013f,0x10010b,0x10010f,0x12010b,0x12010f,0x14c90b,0x14c90f,0x16ca0b,
	0x16ca0f,0x17010b,0x18030f,0x1a060b,0x1a060f,0x1c160f,0x1e0d0b,0x1e0d0f,
	0x20290f,0x22020b,0x22020f,0x24010d,0x28010d,0x2c010d,0x2e010d,0x30010d,
	0x32010d,0x34010d,0x36010b,0x36010f,0x38010f,0x3a010f,0x3c0109,0x3c010d,
	0x3e0109,0x3e010d,0x40010d,0x400109,0x42010d,0x440109,0x44010d,0x46010d,
	0x48010d,0x4a010f,0x4c010f,0x4e010b,0x4f010b,0x50010b,0x50010f,0x52010b,
	0x52010f,0x54010b,0x56010b,0x58010b,0x58010f,0x5a010b,0x5a010f,0x5c010b,
	0x5c010f,0x5e010b,0x5e010f,0x60010b,0x62010b,0x64010b,0x66010b,0x68010f,
	0x68010b,0x6a010f,0x6a010b,0x6c010b,0x6e010b,0x6f010b,0x70010b,0x72010b,
	0x72010f,0x74010b,0x74010f,0x76010f,0x78010f,0x7a010f,0x7c010b,0x7c010f,
	0x80010b,0x80010f,0x82010b,0x82010f,0x84010f,0x86010f,0x88010f,0x8a010f,
	0x8c010b,0x8e010f,0x90010f,0x92010f,0x94010d,0x960109,0x96010d,0x980109,
	0x98010d,0x9a010d,0x9c0109,0x9c010d,0x9e010b,0x9e010f,0x100010b,0x100010f,
	0x102010b,0x102010f,0x104010b,0x104010f,0x106010b,0x106010f,0x107010f,0x108010f,
	0x109010f,0x10a010f,0x10b010f,0x10c010f,0x10d010f,0x10e010f,0x10f010f,0x110010f,
	0x111010f,0x112010f,0x113010f,0x114010f,0x115010f,0x116010f,0x117010f,0x118010f,
	0x119010f,0x11a010f,0x11b010f,0x11c010f,0x11d010f,0x11e010f,0x11f010f,0x120010f,
	0x121010f,0x122010f,0x123010f,0x124010f,0x125010f,0x126010f,0x127010f,0x128010f,
	0x129010f,0x12a010f,0x12b010f,0x12c010f,0x12d010f,0x12e010f,0x12f010f,0x130010f,
	0x131010f,0x132010f,0x133010f,0x134010f,0x135010f,0x136010f,0x137010f,0x138010f,
	0x139010f,0x13b010f,0x13c010d,0x13d010d,0x13e010f,0x13f010f,0x140010f,0x141010f,
	0x142010f,0x143010d,0x144010d,0x145010d,0x146010d,0x147010d,0x148010d,0x149010d,
	0x14a010d,0x14b010d,0x14c010d,0x14d010d,0x14e010d,0x14f010d,0x150010d,0x151010d,
	0x152010d,0x153010d,0x154010d,0x155010d,0x156010d,0x157010d,0x158010d,0x159010d,
	0x15a010d,0x15b010d,0x15c010d,0x15d010d,0x15e010d,0x15f010d,0x160010f,0x161010f,
	0x162010f,0x163010f,0x164010f,0x165010f,0x166010f,0x167010f,0x168010f,0x169010f,
	0x16a010f,0x16b010f,0x16c010f,0x16d010f,0x16e010f,0x16f010f,0x170010f,0x171010f,
	0x172010f,0x174010f,0x175010f,0x176010f,0x177010f,0x178010f,0x179010f,0x17a010d,
	0x17b010d,0x17c010d,0x17d0119,0x17e0119,0x17f0119,0x1800119,0x1810119,0x182010d,
	0x183010d,0x184010d,0x185010d,0x186010d,0x187010d,0x189010f,0x18a010f,0x18b010f,
	0x18c010f,0x18d0111,0x18e0111,0x18f0111,0x1900111,0x1910111,0x1920111,0x1930111,
	0x1940111,0x1950111,0x1960111,0x197010f,0x198010f,0x199010b,0x199010f,0x6c50109,
	0x6c60109,0x6c70109,0x6c80109,0x6c90109,0x6ca0109,0x6cb0109,0x6cc0109,0x6cd0109,
	0x6ce0109,0x6cf0109,0x6cf010d,0x6cf0121,0x6cf0125,0x6e30109,0x6e3010d,0x6e30121,
	0x6e30125,0x6fb0109,0x6fb010d,0x6fb0121,0x6fb0125,0x7070109,0x707010d,0x7070121,
	0x7070125,0x72f0109,0x72f010d,0x72f0121,0x72f0125,0x75b0109,0x75b010d,0x75b0121,
	0x75b0125,0x7630109,0x763010d,0x7630121,0x7630125,0x7830109,0x783010d,0x7830121,
	0x7830125,0x79d0109,0x79d010d,0x79d0121,0x79d0125,0x7b10109,0x7b1010d,0x7b10121,
	0x7b10125,0x7c10109,0x7c1010d,0x7c10121,0x7c10125,0x7cd0109,0x7cd010d,0x7cd0121,
	0x7cd0125,0x7f50109,0x7f5010d,0x7f50121,0x7f50125,0x7fd0109,0x7fd010d,0x7fd0121,
	0x7fd0125,0x8150109,0x815010d,0x8150121,0x8150125,0x82d0109,0x82d010d,0x82d0121,
	0x82d0125,0x8390109,0x839010d,0x8390121,0x8390125,0x8450109,0x845010d,0x8450121,
	0x8450125,0x86d0109,0x86d010d,0x86d0121,0x86d0125,0x8890109,0x889010d,0x8890121,
	0x8890125,0x8ad0109,0x8ad010d,0x8ad0121,0x8ad0125,0x8c50109,0x8c5010d,0x8c50121,
	0x8c50125,0x8d10109,0x8d1010d,0x8d10121,0x8d10125,0x8d90109,0x8d9010d,0x8d90121,
	0x8d90125,0x8dd0109,0x8dd010d,0x8dd0121,0x8dd0125,0x8e90109,0x8e9010d,0x8e90121,
	0x8e90125,0x10200139,0x10200135,0x10200149,0x10200141,0x10200145,0x1020013d,0x10260139,
	0x10260135,0x10260149,0x10260141,0x10260145,0x1026013d,0x102c0139,0x102c0135,0x102c0149,
	0x102c0141,0x102c0145,0x102c013d,0x10330139,0x10330135,0x10330149,0x10330141,0x10330145,
	0x1033013d,0x10390139,0x10390135,0x10390149,0x10390141,0x10390145,0x1039013d,0x103f0135,
	0x103f0145,0x103f013d,0x10440135,0x10440145,0x1044013d,0x10490135,0x10490145,0x1049013d,
	0x104e0135,0x104e0145,0x104e013d,0x10530135,0x10530145,0x1053013d,0x10580135,0x10580145,
	0x1058013d,0x105d0135,0x105d0145,0x105d013d,0x10620135,0x10620145,0x1062013d,0x10670135,
	0x10670145,0x1067013d,0x106c0135,0x106c0145,0x106c013d,0x10710135,0x10710145,0x1071013d,
	0x10760135,0x10760145,0x1076013d,0x107b0139,0x107b0135,0x107b0149,0x107b0141,0x107b0145,
	0x107b013d,0x10830135,0x10830145,0x1083013d,0x10880135,0x10880145,0x1088013d,0x108d0135,
	0x108d0145,0x108d013d,0x10900135,0x10900145,0x1090013d,0x10930135,0x10930145,0x1093013d,
	0x10960135,0x10960145,0x1096013d,0x10990135,0x10990145,0x1099013d,0x109c0135,0x109c0145,
	0x109c013d,0x10a30135,0x10a30145,0x10a3013d,0x10aa0135,0x10aa0145,0x10aa013d,0x10b10135,
	0x10b10145,0x10b1013d,0x10b80135,0x10b80145,0x10b8013d,0x10bf0135,0x10bf0145,0x10bf013d,
	0x10c20135,0x10c20145,0x10c2013d,0x10c50135,0x10c50145,0x10c5013d,0x10c80135,0x10c80145,
	0x10c8013d,0x10cb0135,0x10cb0145,0x10cb013d,0x10ce0139,0x10ce0135,0x10ce0149,0x10ce0141,
	0x10ce0145,0x10ce013d,0x10d40139,0x10d40135,0x10d40149,0x10d40141,0x10d40145,0x10d4013d,
	0x10da0139,0x10da0135,0x10da0149,0x10da0141,0x10da0145,0x10da013d,0x10e00135,0x10e00145,
	0x10e0013d,0x10e30135,0x10e30145,0x10e3013d,0x10e60135,0x10e60145,0x10e6013d,0x10e90135,
	0x10e90145,0x10e9013d,0x10ec0135,0x10ec0145,0x10ec013d,0x10ef0139,0x10ef0135,0x10ef0149,
	0x10ef0145,0x10ef013d,0x10f40135,0x10f40145,0x10f60135,0x10f60145,0x10f80135,0x10f80145,
	0x10f8013d,0x10fb0135,0x10fb0145,0x10fb013d,
	};

static const TUint32 TheKanaIndex[] = 
	{
	0x210013,0x22010d,0x230076,0x24006f,0x250074,0x260078,0x270019,0x28003d,
	0x29003f,0x2a007a,0x2b0057,0x2c0006,0x2d0059,0x2e0008,0x2f002e,0x30010f,
	0x310110,0x320111,0x330112,0x340113,0x350114,0x360115,0x370116,0x380117,
	0x390118,0x3a000c,0x3b000e,0x3c0060,0x3d005e,0x3e0062,0x3f0011,0x40007c,
	0x41011b,0x42011f,0x430123,0x440127,0x45012b,0x46012f,0x470133,0x480137,
	0x49013b,0x4a013f,0x4b0143,0x4c0147,0x4d014b,0x4e014f,0x4f0153,0x500157,
	0x51015b,0x52015f,0x530163,0x540167,0x55016b,0x56016f,0x570173,0x580177,
	0x59017b,0x5a017f,0x5b0043,0x5c0030,0x5d0045,0x5e001e,0x5f0021,0x60001b,
	0x610119,0x62011d,0x630121,0x640125,0x650129,0x66012d,0x670131,0x680135,
	0x690139,0x6a013d,0x6b0141,0x6c0145,0x6d0149,0x6e014d,0x6f0151,0x700155,
	0x710159,0x72015d,0x730161,0x740165,0x750169,0x76016d,0x770171,0x780175,
	0x790179,0x7a017d,0x7b0047,0x7c0035,0x7d0049,0x7e0033,0xa20071,0xa30072,
	0xa5006d,0xa7007e,0xa8001d,0xb00069,0xb1005b,0xb4001a,0xb600b7,0xd7005c,
	0xf7005d,0x2010002d,0x2015002c,0x20180039,0x2019003a,0x201c003b,0x201d003c,0x202000b5,
	0x202100b6,0x20250038,0x20260037,0x203000b1,0x2032006a,0x2033006b,0x203b008c,0x2103006c,
	0x211600ef,0x212100f1,0x216000cc,0x216100cd,0x216200ce,0x216300cf,0x216400d0,0x216500d1,
	0x216600d2,0x216700d3,0x216800d4,0x216900d5,0x21700101,0x21710102,0x21720103,0x21730104,
	0x21740105,0x21750106,0x21760107,0x21770108,0x21780109,0x2179010a,0x2190008f,0x21910090,
	0x2192008e,0x21930091,0x21d2009e,0x21d4009f,0x220000a0,0x220200a5,0x220300a1,0x220700a6,
	0x22080093,0x220b0094,0x221100fe,0x2212005a,0x221a00ab,0x221d00ad,0x221e0066,0x221f00ff,
	0x222000a2,0x22250034,0x2227009b,0x2228009c,0x2229009a,0x222a0099,0x222b00af,0x222c00b0,
	0x222e00fd,0x22340066,0x223500ae,0x223d00ac,0x225200a8,0x226100a7,0x22660064,0x22670065,
	0x226a00a9,0x226b00aa,0x22820097,0x22830098,0x22860095,0x22870096,0x22a500a3,0x22bf0100,
	0x231200a4,0x246000b9,0x246100ba,0x246200bb,0x246300bc,0x246400bd,0x246500be,0x246600bf,
	0x246700c0,0x246800c1,0x246900c2,0x246a00c3,0x246b00c3,0x246c00c4,0x246d00c5,0x246e00c6,
	0x246f00c7,0x247000c8,0x247100c9,0x247200ca,0x247300cb,0x25a00087,0x25a10086,0x25b20089,
	0x25b30088,0x25bc008b,0x25bd008a,0x25c60085,0x25c70084,0x25cb0081,0x25ce0083,0x25cf0082,
	0x25ef00b8,0x26050080,0x2606007f,0x26400068,0x26420067,0x266a00b4,0x266d00b3,0x266f00b2,
	0x30010003,0x30020005,0x30030025,0x30050027,0x30060028,0x30070029,0x3008004b,0x3009004c,
	0x300a004d,0x300b004e,0x300c004f,0x300d0051,0x300e0053,0x300f0054,0x30100055,0x30110056,
	0x3012008d,0x30130092,0x30140041,0x30150042,0x301c0032,0x301d00ed,0x301f00ee,0x30410181,
	0x30420182,0x30430187,0x30440188,0x3045018d,0x3046018e,0x30470193,0x30480194,0x30490199,
	0x304a019a,0x304b019f,0x304d01a2,0x304f01a5,0x305101a8,0x305301ab,0x305501ae,0x305701b1,
	0x305901b4,0x305b01b7,0x305d01ba,0x305f01bd,0x306101c0,0x306301c3,0x306401c4,0x306601c9,
	0x306801cc,0x306a01cf,0x306b01d2,0x306c01d5,0x306d01d8,0x306e01db,0x306f01de,0x307201e1,
	0x307501e4,0x307801e7,0x307b01ea,0x307e01ed,0x307f01f0,0x308001f3,0x308101f6,0x308201f9,
	0x308301fc,0x308401fd,0x30850202,0x30860203,0x30870208,0x30880209,0x3089020e,0x308a0211,
	0x308b0214,0x308c0217,0x308d021a,0x308e021d,0x308f021e,0x30900222,0x30910224,0x30920226,
	0x30930229,0x30990000,0x309a0001,0x309b0016,0x309c0018,0x309d0024,0x30a10183,0x30a20185,
	0x30a30189,0x30a4018b,0x30a5018f,0x30a60191,0x30a70195,0x30a80197,0x30a9019b,0x30aa019d,
	0x30ab01a0,0x30ad01a3,0x30af01a6,0x30b101a9,0x30b301ac,0x30b501af,0x30b701b2,0x30b901b5,
	0x30bb01b8,0x30bd01bb,0x30bf01be,0x30c101c1,0x30c301c5,0x30c401c7,0x30c601ca,0x30c801cd,
	0x30ca01d0,0x30cb01d3,0x30cc01d6,0x30cd01d9,0x30ce01dc,0x30cf01df,0x30d201e2,0x30d501e5,
	0x30d801e8,0x30db01eb,0x30de01ee,0x30df01f1,0x30e001f4,0x30e101f7,0x30e201fa,0x30e301fe,
	0x30e40200,0x30e50204,0x30e60206,0x30e7020a,0x30e8020c,0x30e9020f,0x30ea0212,0x30eb0215,
	0x30ec0218,0x30ed021b,0x30ee021f,0x30ef0220,0x30f00223,0x30f10225,0x30f20227,0x30f3022a,
	0x30fb000a,0x30fc002b,0x30fd0023,0x323100f7,0x323200f8,0x323900f9,0x32a400f2,0x32a500f3,
	0x32a600f4,0x32a700f5,0x32a800f6,0x330300dc,0x330d00e0,0x331400d7,0x331800da,0x332200d8,
	0x332300e2,0x332600e1,0x332700db,0x332b00e3,0x333600dd,0x333b00e5,0x334900d6,0x334a00e4,
	0x334d00d9,0x335100de,0x335700df,0x337b00ec,0x337c00fc,0x337d00fb,0x337e00fa,0x338f00e9,
	0x339c00e6,0x339d00e7,0x339e00e8,0x33a100eb,0x33c400ea,0x33cd00f0,0x4edd0026,0xfe540010,
	0xff010014,0xff02010e,0xff030077,0xff040070,0xff050075,0xff060079,0xff07010c,0xff08003e,
	0xff090040,0xff0a007b,0xff0b0058,0xff0c0007,0xff0d005a,0xff0e0009,0xff0f002f,0xff1a000d,
	0xff1b000f,0xff1c0061,0xff1d005f,0xff1e0063,0xff1f0012,0xff20007d,0xff21011c,0xff220120,
	0xff230124,0xff240128,0xff25012c,0xff260130,0xff270134,0xff280138,0xff29013c,0xff2a0140,
	0xff2b0144,0xff2c0148,0xff2d014c,0xff2e0150,0xff2f0154,0xff300158,0xff31015c,0xff320160,
	0xff330164,0xff340168,0xff35016c,0xff360170,0xff370174,0xff380178,0xff39017c,0xff3a0180,
	0xff3b0044,0xff3c0031,0xff3d0046,0xff3e001f,0xff3f0022,0xff40001c,0xff41011a,0xff42011e,
	0xff430122,0xff440126,0xff45012a,0xff46012e,0xff470132,0xff480136,0xff49013a,0xff4a013e,
	0xff4b0142,0xff4c0146,0xff4d014a,0xff4e014e,0xff4f0152,0xff500156,0xff51015a,0xff52015e,
	0xff530162,0xff540166,0xff55016a,0xff56016e,0xff570172,0xff580176,0xff59017a,0xff5a017e,
	0xff5b0048,0xff5c0036,0xff5d004a,0xff5e0032,0xff610004,0xff620050,0xff630052,0xff640002,
	0xff65000b,0xff660228,0xff670184,0xff68018a,0xff690190,0xff6a0196,0xff6b019c,0xff6c01ff,
	0xff6d0205,0xff6e020b,0xff6f01c6,0xff70002a,0xff710186,0xff72018c,0xff730192,0xff740198,
	0xff75019e,0xff7601a1,0xff7701a4,0xff7801a7,0xff7901aa,0xff7a01ad,0xff7b01b0,0xff7c01b3,
	0xff7d01b6,0xff7e01b9,0xff7f01bc,0xff8001bf,0xff8101c2,0xff8201c8,0xff8301cb,0xff8401ce,
	0xff8501d1,0xff8601d4,0xff8701d7,0xff8801da,0xff8901dd,0xff8a01e0,0xff8b01e3,0xff8c01e6,
	0xff8d01e9,0xff8e01ec,0xff8f01ef,0xff9001f2,0xff9101f5,0xff9201f8,0xff9301fb,0xff940201,
	0xff950207,0xff96020d,0xff970210,0xff980213,0xff990216,0xff9a0219,0xff9b021c,0xff9c0221,
	0xff9d022b,0xff9e0015,0xff9f0017,0xffe00071,0xffe10073,0xffe2009d,0xffe30020,0xffe4010b,
	0xffe5006e,
	};

/*
Collation keys for the Hiragana and Katakana syllabaries, built by
running coltab on \loce32\japanese\source\japanese_basekeys.txt after deleting all CJK characters.
*/
static const TCollationKeyTable TheKanaTable = 
	{ TheKanaKey, TheKanaIndex, 561, NULL, NULL, 0 };

static const TCollationMethod TheKanaMethod =
	{ 0, NULL, &TheKanaTable, 0 };

static const TCollationMethod TheSwapKanaMethod =
	{ 0, NULL, &TheKanaTable, TCollationMethod::ESwapKana };

static const TUint32 TheQazKey[] = 
	{
	0x6d00121,0x6e40121,
	};

static const TUint32 TheQazIndex[] = 
	{
	0
	};

static const TUint16 TheQazStringElement[] = 
	{
	0x2,0x53,0x45,0x3,0x53,0x45,0x58,
	};

static const TUint32 TheQazStringIndex[] = 
	{
	0x0,0x30001,
	};

static const TCollationKeyTable TheQazTable = 
	{ TheQazKey, TheQazIndex, 0, TheQazStringElement, TheQazStringIndex, 2 };

// A bizarre collation method that sorts 'SE' between A and B and 'SEX' between B and C
static const TCollationMethod TheQazMethod = 
	{ 0, NULL, &TheQazTable, 0 };

// A collation method to sort accents starting at the end of the string, as in France
static const TCollationMethod TheAccentsBackwardsMethod = 
	{ 0, NULL, NULL, TCollationMethod::EAccentsBackwards };

void test_unicode_collations()
	{
	/*
	Note: the curious method of getting values greater than 255 into the strings is needed
	because GCC doesn't handle the \x notation properly: it zeroes the high byte, so
	L"\x2011" puts the value 0x0011, not 0x2011, into the string.
	*/
	TBuf<32> x, y;

	test.Next(_L("collation"));
	test_unicode_collation(_L("SEZ"),_L("BZ"),-1,3,&TheQazMethod); // test 2-character combination 'SE'
	test_unicode_collation(_L("SEXY"),_L("CY"),-1,3,&TheQazMethod); // test 3-character combination 'SEX'
	test_unicode_collation(_L("Graham"),_L("Graham"),0);		// equality
	test_unicode_collation(_L("something"),_L(""),1);			// something after nothing
	test_unicode_collation(_L("inform"),_L("information"),-1);	// prefixes come first
	test_unicode_collation(_L("unicode"),_L("consortium"),1);	// standard alphabetic ordering: lowercase
	test_unicode_collation(_L("AARON"),_L("AARDVARK"),1);		// standard alphabetic ordering: uppercase
	test_unicode_collation(_L("AARON"),_L("AARDVARK"),1,0);		// alphabetic ordering is significant at level 0
	test_unicode_collation(_L("hello"),_L("Hello"),-1,3);		// lowercase before uppercase
	test_unicode_collation(_L("hello"),_L("Hello"),-1,3,&TheSwapKanaMethod);// swapping kana ordering doesn't affect case
	test_unicode_collation(_L("hello"),_L("Hello"),1,3,&TheSwapCaseMethod); // uppercase before lowercase if ESwapCase used
	test_unicode_collation(_L("hello"),_L("Hello"),0,2,&TheFoldCaseMethod); // uppercase = lowercase if EFoldCase used at level 2
	test_unicode_collation(_L("hello"),_L("Hello"),-1,2);		// case is significant at level 2
	test_unicode_collation(_L("hello"),_L("Hello"),0,1);		// ignore case at level 1
	test_unicode_collation(_L("hello"),_L("[H E L L O !]"),0,1);// ignore punctuation and case at level 1
	test_unicode_collation(_L("1234"),_L("abc"),-1);			// digits before letters
	x = _L("week day"); x[4] = 0x2011;							// (week-day with non-breaking hyphen)
	test_unicode_collation(x,_L("weekend"),-1);					// ignore punctuation if letters differ; d < e
	test_unicode_collation(x,_L("week-day"),1);					// compare punctuation if otherwise equal
	test_unicode_collation(x,_L("week-day"),0,2);				// ignore difference between hyphens at level 2
	test_unicode_collation(x,_L("Week-day"),-1);				// case difference outweighs punctuation ordering
	y = _L("a ngstro m"); y[1] = 0x30a; y[8] = 0x308;			// (a<ring>ngstro<umlaut>m with piece accents)
	test_unicode_collation(_L("\xe5ngstr\xf6m"),y,0);			// base+accent = precomposed (a<ring>ngstro<umlaut>m)
	test_unicode_collation(_L("Liege"),_L("Li\xe8ge"),-1);		// unaccented before accented...
	test_unicode_collation(_L("Liege"),_L("Li\xe8ge"),-1,1);	// accents are significant at level 1
	test_unicode_collation(_L("Liege"),_L("li\xe8ge"),-1);		// accentuation outweighs case
	test_unicode_collation(_L("Liege"),_L("li\xe8ge"),0,0);		// ignore accents and case at level 0
	test_unicode_collation(_L("\xe5ngstr\xf6m"),_L("zoo"),-1);	// a-ring before z by default
	test_unicode_collation(_L("\xe5ngstr\xf6m"),_L("zoo"),1,3,&TheSwedishMethod);	// a-ring after z in Sweden
	test_unicode_collation(_L("Antidisestablishmentarian"),_L("antidisestablishmentariac"),1);
																// long strings that don't have identical prefixes
	x = _L("\x304b\x306a"); y = _L("\x30ab\x30ca");				// 'kana' in Hiragana and Katakana
	test_unicode_collation(x,y,-1,3,&TheKanaMethod);			// Hiragana before Katakana 
	test_unicode_collation(x,y,1,3,&TheSwapKanaMethod);			// Katakana before Hiragana if ESwapKana used
	x = _L("p\xe9"L"ch\xe9"); y = _L("p\xea"L"che");			// x = pe<acute>che<acute>, y = pe<circumflex>che
	test_unicode_collation(x,y,-1,3);							// normal order; accents compared from start
	test_unicode_collation(x,y,1,3,&TheAccentsBackwardsMethod);	// reverse order; accents compared from end
	test_unicode_collation(_L("aaAA"),_L("aAaA"),0,3,&TheFoldCaseMethod); // case folding should work even at level 3
	test_unicode_collation(_L("\x2113"),_L("L"),1,3,&TheFoldCaseMethod); // script l and L should not be equal;
																// script l before L because L folds to lower case,
																// which sorts after secondary key type for script l
	test_unicode_collation(_L("\x3c2"),_L("\x3c3"),-1,3); // Greek final and non-final sigma
	test_unicode_collation(_L("\x3c2"),_L("\x3c3"),0,3,&TheFoldCaseMethod); // Greek final and non-final sigma
	test_unicode_collation(_L("\xee0"),_L("\x4e00"),1,3);		// undefined character after CJK character not in table
	test_unicode_collation(_L("\xee0"),_L("\xef0"),-1,3);		// undefined characters must differ
	test_unicode_collation(_L("\xee0\xee1"),_L("\xef0"),-1,3);	// undefined characters in Unicode order, not string length
	test_unicode_collation(_L("\xd800\xdc00"),_L("\xceff"),1,3);	// character U+10000 (using surrogates) 
																// after Hangul character not in table
	}

// range of characters newly added or changed in Unicode 5.0
// Inside BMP
static const TUint32 Unicode5NewCharRange[] = 
	{
	0x0220, 0x0221, 0x0234, 0x024F, 0x02AE, 0x02AF, 0x02EF, 0x02FF,
	0x034F, 0x035F, 0x0363, 0x036F, 0x037B, 0x037D, 0x03D8, 0x03D9,
	0x03F4, 0x03FF, 0x048A, 0x048B, 0x04C5, 0x04C6, 0x04C9, 0x04CA,
	0x04CD, 0x04CF, 0x04F6, 0x0513, 0x05A2, 0x05A2, 0x05BA, 0x05BA,
	0x05C5, 0x05C7, 0x0600, 0x0603, 0x060B, 0x0615, 0x061E, 0x061F,
	0x0656, 0x065E, 0x066E, 0x066F, 0x06EE, 0x06EF, 0x06FF, 0x06FF,
	0x072D, 0x072F, 0x074D, 0x076D, 0x07B1, 0x07B1, 0x07C0, 0x07FA,
	0x0904, 0x0904, 0x097B, 0x097F, 0x09BD, 0x09BD, 0x09CE, 0x09CE,
	0x0A01, 0x0A03, 0x0A8C, 0x0A8C, 0x0AE1, 0x0AE3, 0x0AF1, 0x0AF1,
	0x0B35, 0x0B35, 0x0B71, 0x0B71, 0x0BB6, 0x0BB6, 0x0BE6, 0x0BE6,
	0x0BF3, 0x0BFA, 0x0CBC, 0x0CBD, 0x0CE2, 0x0CE3, 0x0CF1, 0x0CF2,
	0x0FD0, 0x0FD1, 0x10F7, 0x10FC, 0x1207, 0x1207, 0x1247, 0x1247,
	0x1287, 0x1287, 0x12AF, 0x12AF, 0x12CF, 0x12CF, 0x12EF, 0x12EF,
	0x130F, 0x130F, 0x131F, 0x131F, 0x1347, 0x1347, 0x135F, 0x1360,
	0x1380, 0x1399, 0x1700, 0x170C, 0x170E, 0x1714, 0x1720, 0x1736, 
	0x1740, 0x1753,	0x1760, 0x176C, 0x176E, 0x1770, 0x1772, 0x1773, 
	0x17DD, 0x17DD, 0x17F0, 0x17F9, 0x1900, 0x191C, 0x1920, 0x192B, 
	0x1930, 0x193B, 0x1940, 0x1940, 0x1944, 0x196D, 0x1970, 0x1974,
	0x1980, 0x19A9, 0x19B0, 0x19C9, 0x19D0, 0x19D9, 0x19DE, 0x1A1B,
	0x1A1E, 0x1A1F, 0x1B00, 0x1B4B, 0x1B50, 0x1B7C, 0x1D00, 0x1DCA, 
	0x1DFE, 0x1DFF, 0x2047, 0x2063, 0x2071, 0x2071, 0x2090, 0x2094, 
	0x20B0, 0x20B5, 0x20E4, 0x20EF, 0x213B, 0x214E, 0x2184, 0x2184,
 	0x21F4, 0x21FF, 0x22F2, 0x22FF, 0x237C, 0x237C, 0x239B, 0x23E7, 
	0x24EB, 0x24FF, 0x2596, 0x259F, 0x25F8, 0x25FF, 0x2614, 0x2618, 
	0x2672, 0x269C, 0x26A0, 0x26B2, 0x2768, 0x2775, 0x27C0, 0x27CA,
 	0x27D0, 0x27EB, 0x27F0, 0x27FF, 0x2900, 0x2B1A, 0x2B20, 0x2B23, 
	0x2C00, 0x2C2E, 0x2C30, 0x2C5E, 0x2C60, 0x2C6C, 0x2C74, 0x2C77,
 	0x2C80, 0x2CEA, 0x2CF9, 0x2D25, 0x2D30, 0x2D65, 0x2D6F, 0x2D6F,
 	0x2D80, 0x2D96, 0x2DA0, 0x2DA6, 0x2DA8, 0x2DAE, 0x2DB0, 0x2DB6, 
	0x2DB8, 0x2DBE, 0x2DC0, 0x2DC6, 0x2DC8, 0x2DCE, 0x2DD0, 0x2DD6, 
	0x2DD8, 0x2DDE, 0x2E00, 0x2E17, 0x2E1C, 0x2E1D, 0x303B, 0x303D,
	0x3095, 0x3096, 0x309F, 0x30A0, 0x30FF, 0x30FF, 0x31C0, 0x31CF,
	0x31F0, 0x31FF, 0x321D, 0x321E, 0x3250, 0x325F, 0x327C, 0x327E,
	0x32B1, 0x32BF, 0x32CC, 0x32CF, 0x3377, 0x337A, 0x33DE, 0x33DF,
	0x33FF, 0x33FF, 0x4DC0, 0x4DFF, 0x9FA6, 0x9FBB, 0xA4A2, 0xA4A3,
	0xA4B4, 0xA4B4, 0xA4C1, 0xA4C1, 0xA4C5, 0xA4C5, 0xA700, 0xA71A,
	0xA720, 0xA721, 0xA800, 0xA82B, 0xA840, 0xA877, 0xFA30, 0xFA6A,
	0xFA70, 0xFAD9, 0xFDFC, 0xFDFD, 0xFE00, 0xFE19, 0xFE45, 0xFE48,
	0xFE73, 0xFE73, 0xFF5F, 0xFF60, 
	};

// characters Codepoint out of BMP, newly added or changed in Unicode 5.0
static const TUint32 Unicode5NewCharOutBMPIndex[] =
{
	0x10000, 0x10001, 0x10002, 0x10003, 0x10004, 0x10005, 0x10006, 0x10007, 
	0x10008, 0x10009, 0x1000A, 0x1000B, 0x1000D, 0x1000E, 0x1000F, 0x10010, 
	0x10011, 0x10012, 0x10013, 0x10014, 0x10015, 0x10016, 0x10017, 0x10018, 
	0x10019, 0x1001A, 0x1001B, 0x1001C, 0x1001D, 0x1001E, 0x1001F, 0x10020, 
	0x10021, 0x10022, 0x10023, 0x10024, 0x10025, 0x10026, 0x10028, 0x10029, 
	0x1002A, 0x1002B, 0x1002C, 0x1002D, 0x1002E, 0x1002F, 0x10030, 0x10031, 
	0x10032, 0x10033, 0x10034, 0x10035, 0x10036, 0x10037, 0x10038, 0x10039, 
	0x1003A, 0x1003C, 0x1003D, 0x1003F, 0x10040, 0x10041, 0x10042, 0x10043, 
	0x10044, 0x10045, 0x10046, 0x10047, 0x10048, 0x10049, 0x1004A, 0x1004B, 
	0x1004C, 0x1004D, 0x10050, 0x10051, 0x10052, 0x10053, 0x10054, 0x10055, 
	0x10056, 0x10057, 0x10058, 0x10059, 0x1005A, 0x1005B, 0x1005C, 0x1005D, 
	0x10080, 0x10081, 0x10082, 0x10083, 0x10084, 0x10085, 0x10086, 0x10087, 
	0x10088, 0x10089, 0x1008A, 0x1008B, 0x1008C, 0x1008D, 0x1008E, 0x1008F, 
	0x10090, 0x10091, 0x10092, 0x10093, 0x10094, 0x10095, 0x10096, 0x10097, 
	0x10098, 0x10099, 0x1009A, 0x1009B, 0x1009C, 0x1009D, 0x1009E, 0x1009F, 
	0x100A0, 0x100A1, 0x100A2, 0x100A3, 0x100A4, 0x100A5, 0x100A6, 0x100A7, 
	0x100A8, 0x100A9, 0x100AA, 0x100AB, 0x100AC, 0x100AD, 0x100AE, 0x100AF, 
	0x100B0, 0x100B1, 0x100B2, 0x100B3, 0x100B4, 0x100B5, 0x100B6, 0x100B7, 
	0x100B8, 0x100B9, 0x100BA, 0x100BB, 0x100BC, 0x100BD, 0x100BE, 0x100BF, 
	0x100C0, 0x100C1, 0x100C2, 0x100C3, 0x100C4, 0x100C5, 0x100C6, 0x100C7, 
	0x100C8, 0x100C9, 0x100CA, 0x100CB, 0x100CC, 0x100CD, 0x100CE, 0x100CF, 
	0x100D0, 0x100D1, 0x100D2, 0x100D3, 0x100D4, 0x100D5, 0x100D6, 0x100D7, 
	0x100D8, 0x100D9, 0x100DA, 0x100DB, 0x100DC, 0x100DD, 0x100DE, 0x100DF, 
	0x100E0, 0x100E1, 0x100E2, 0x100E3, 0x100E4, 0x100E5, 0x100E6, 0x100E7, 
	0x100E8, 0x100E9, 0x100EA, 0x100EB, 0x100EC, 0x100ED, 0x100EE, 0x100EF, 
	0x100F0, 0x100F1, 0x100F2, 0x100F3, 0x100F4, 0x100F5, 0x100F6, 0x100F7, 
	0x100F8, 0x100F9, 0x100FA, 0x10100, 0x10101, 0x10102, 0x10107, 0x10108, 
	0x10109, 0x1010A, 0x1010B, 0x1010C, 0x1010D, 0x1010E, 0x1010F, 0x10110, 
	0x10111, 0x10112, 0x10113, 0x10114, 0x10115, 0x10116, 0x10117, 0x10118, 
	0x10119, 0x1011A, 0x1011B, 0x1011C, 0x1011D, 0x1011E, 0x1011F, 0x10120, 
	0x10121, 0x10122, 0x10123, 0x10124, 0x10125, 0x10126, 0x10127, 0x10128, 
	0x10129, 0x1012A, 0x1012B, 0x1012C, 0x1012D, 0x1012E, 0x1012F, 0x10130, 
	0x10131, 0x10132, 0x10133, 0x10137, 0x10138, 0x10139, 0x1013A, 0x1013B, 
	0x1013C, 0x1013D, 0x1013E, 0x1013F, 0x10140, 0x10141, 0x10142, 0x10143, 
	0x10144, 0x10145, 0x10146, 0x10147, 0x10148, 0x10149, 0x1014A, 0x1014B, 
	0x1014C, 0x1014D, 0x1014E, 0x1014F, 0x10150, 0x10151, 0x10152, 0x10153, 
	0x10154, 0x10155, 0x10156, 0x10157, 0x10158, 0x10159, 0x1015A, 0x1015B, 
	0x1015C, 0x1015D, 0x1015E, 0x1015F, 0x10160, 0x10161, 0x10162, 0x10163, 
	0x10164, 0x10165, 0x10166, 0x10167, 0x10168, 0x10169, 0x1016A, 0x1016B, 
	0x1016C, 0x1016D, 0x1016E, 0x1016F, 0x10170, 0x10171, 0x10172, 0x10173, 
	0x10174, 0x10175, 0x10176, 0x10177, 0x10178, 0x10179, 0x1017A, 0x1017B, 
	0x1017C, 0x1017D, 0x1017E, 0x1017F, 0x10180, 0x10181, 0x10182, 0x10183, 
	0x10184, 0x10185, 0x10186, 0x10187, 0x10188, 0x10189, 0x1018A, 0x10300, 
	0x10301, 0x10302, 0x10303, 0x10304, 0x10305, 0x10306, 0x10307, 0x10308, 
	0x10309, 0x1030A, 0x1030B, 0x1030C, 0x1030D, 0x1030E, 0x1030F, 0x10310, 
	0x10311, 0x10312, 0x10313, 0x10314, 0x10315, 0x10316, 0x10317, 0x10318, 
	0x10319, 0x1031A, 0x1031B, 0x1031C, 0x1031D, 0x1031E, 0x10320, 0x10321, 
	0x10322, 0x10323, 0x10330, 0x10331, 0x10332, 0x10333, 0x10334, 0x10335, 
	0x10336, 0x10337, 0x10338, 0x10339, 0x1033A, 0x1033B, 0x1033C, 0x1033D, 
	0x1033E, 0x1033F, 0x10340, 0x10341, 0x10342, 0x10343, 0x10344, 0x10345, 
	0x10346, 0x10347, 0x10348, 0x10349, 0x1034A, 0x10380, 0x10381, 0x10382, 
	0x10383, 0x10384, 0x10385, 0x10386, 0x10387, 0x10388, 0x10389, 0x1038A, 
	0x1038B, 0x1038C, 0x1038D, 0x1038E, 0x1038F, 0x10390, 0x10391, 0x10392, 
	0x10393, 0x10394, 0x10395, 0x10396, 0x10397, 0x10398, 0x10399, 0x1039A, 
	0x1039B, 0x1039C, 0x1039D, 0x1039F, 0x103A0, 0x103A1, 0x103A2, 0x103A3, 
	0x103A4, 0x103A5, 0x103A6, 0x103A7, 0x103A8, 0x103A9, 0x103AA, 0x103AB, 
	0x103AC, 0x103AD, 0x103AE, 0x103AF, 0x103B0, 0x103B1, 0x103B2, 0x103B3, 
	0x103B4, 0x103B5, 0x103B6, 0x103B7, 0x103B8, 0x103B9, 0x103BA, 0x103BB, 
	0x103BC, 0x103BD, 0x103BE, 0x103BF, 0x103C0, 0x103C1, 0x103C2, 0x103C3, 
	0x103C8, 0x103C9, 0x103CA, 0x103CB, 0x103CC, 0x103CD, 0x103CE, 0x103CF, 
	0x103D0, 0x103D1, 0x103D2, 0x103D3, 0x103D4, 0x103D5, 0x10400, 0x10401, 
	0x10402, 0x10403, 0x10404, 0x10405, 0x10406, 0x10407, 0x10408, 0x10409, 
	0x1040A, 0x1040B, 0x1040C, 0x1040D, 0x1040E, 0x1040F, 0x10410, 0x10411, 
	0x10412, 0x10413, 0x10414, 0x10415, 0x10416, 0x10417, 0x10418, 0x10419, 
	0x1041A, 0x1041B, 0x1041C, 0x1041D, 0x1041E, 0x1041F, 0x10420, 0x10421, 
	0x10422, 0x10423, 0x10424, 0x10425, 0x10426, 0x10427, 0x10428, 0x10429, 
	0x1042A, 0x1042B, 0x1042C, 0x1042D, 0x1042E, 0x1042F, 0x10430, 0x10431, 
	0x10432, 0x10433, 0x10434, 0x10435, 0x10436, 0x10437, 0x10438, 0x10439, 
	0x1043A, 0x1043B, 0x1043C, 0x1043D, 0x1043E, 0x1043F, 0x10440, 0x10441, 
	0x10442, 0x10443, 0x10444, 0x10445, 0x10446, 0x10447, 0x10448, 0x10449, 
	0x1044A, 0x1044B, 0x1044C, 0x1044D, 0x1044E, 0x1044F, 0x10450, 0x10451, 
	0x10452, 0x10453, 0x10454, 0x10455, 0x10456, 0x10457, 0x10458, 0x10459, 
	0x1045A, 0x1045B, 0x1045C, 0x1045D, 0x1045E, 0x1045F, 0x10460, 0x10461, 
	0x10462, 0x10463, 0x10464, 0x10465, 0x10466, 0x10467, 0x10468, 0x10469, 
	0x1046A, 0x1046B, 0x1046C, 0x1046D, 0x1046E, 0x1046F, 0x10470, 0x10471, 
	0x10472, 0x10473, 0x10474, 0x10475, 0x10476, 0x10477, 0x10478, 0x10479, 
	0x1047A, 0x1047B, 0x1047C, 0x1047D, 0x1047E, 0x1047F, 0x10480, 0x10481, 
	0x10482, 0x10483, 0x10484, 0x10485, 0x10486, 0x10487, 0x10488, 0x10489, 
	0x1048A, 0x1048B, 0x1048C, 0x1048D, 0x1048E, 0x1048F, 0x10490, 0x10491, 
	0x10492, 0x10493, 0x10494, 0x10495, 0x10496, 0x10497, 0x10498, 0x10499, 
	0x1049A, 0x1049B, 0x1049C, 0x1049D, 0x104A0, 0x104A1, 0x104A2, 0x104A3, 
	0x104A4, 0x104A5, 0x104A6, 0x104A7, 0x104A8, 0x104A9, 0x10800, 0x10801, 
	0x10802, 0x10803, 0x10804, 0x10805, 0x10808, 0x1080A, 0x1080B, 0x1080C, 
	0x1080D, 0x1080E, 0x1080F, 0x10810, 0x10811, 0x10812, 0x10813, 0x10814, 
	0x10815, 0x10816, 0x10817, 0x10818, 0x10819, 0x1081A, 0x1081B, 0x1081C, 
	0x1081D, 0x1081E, 0x1081F, 0x10820, 0x10821, 0x10822, 0x10823, 0x10824, 
	0x10825, 0x10826, 0x10827, 0x10828, 0x10829, 0x1082A, 0x1082B, 0x1082C, 
	0x1082D, 0x1082E, 0x1082F, 0x10830, 0x10831, 0x10832, 0x10833, 0x10834, 
	0x10835, 0x10837, 0x10838, 0x1083C, 0x1083F, 0x10900, 0x10901, 0x10902, 
	0x10903, 0x10904, 0x10905, 0x10906, 0x10907, 0x10908, 0x10909, 0x1090A, 
	0x1090B, 0x1090C, 0x1090D, 0x1090E, 0x1090F, 0x10910, 0x10911, 0x10912, 
	0x10913, 0x10914, 0x10915, 0x10916, 0x10917, 0x10918, 0x10919, 0x1091F, 
	0x10A00, 0x10A01, 0x10A02, 0x10A03, 0x10A05, 0x10A06, 0x10A0C, 0x10A0D, 
	0x10A0E, 0x10A0F, 0x10A10, 0x10A11, 0x10A12, 0x10A13, 0x10A15, 0x10A16, 
	0x10A17, 0x10A19, 0x10A1A, 0x10A1B, 0x10A1C, 0x10A1D, 0x10A1E, 0x10A1F, 
	0x10A20, 0x10A21, 0x10A22, 0x10A23, 0x10A24, 0x10A25, 0x10A26, 0x10A27, 
	0x10A28, 0x10A29, 0x10A2A, 0x10A2B, 0x10A2C, 0x10A2D, 0x10A2E, 0x10A2F, 
	0x10A30, 0x10A31, 0x10A32, 0x10A33, 0x10A38, 0x10A39, 0x10A3A, 0x10A3F, 
	0x10A40, 0x10A41, 0x10A42, 0x10A43, 0x10A44, 0x10A45, 0x10A46, 0x10A47, 
	0x10A50, 0x10A51, 0x10A52, 0x10A53, 0x10A54, 0x10A55, 0x10A56, 0x10A57, 
	0x10A58, 0x12000, 0x12001, 0x12002, 0x12003, 0x12004, 0x12005, 0x12006, 
	0x12007, 0x12008, 0x12009, 0x1200A, 0x1200B, 0x1200C, 0x1200D, 0x1200E, 
	0x1200F, 0x12010, 0x12011, 0x12012, 0x12013, 0x12014, 0x12015, 0x12016, 
	0x12017, 0x12018, 0x12019, 0x1201A, 0x1201B, 0x1201C, 0x1201D, 0x1201E, 
	0x1201F, 0x12020, 0x12021, 0x12022, 0x12023, 0x12024, 0x12025, 0x12026, 
	0x12027, 0x12028, 0x12029, 0x1202A, 0x1202B, 0x1202C, 0x1202D, 0x1202E, 
	0x1202F, 0x12030, 0x12031, 0x12032, 0x12033, 0x12034, 0x12035, 0x12036, 
	0x12037, 0x12038, 0x12039, 0x1203A, 0x1203B, 0x1203C, 0x1203D, 0x1203E, 
	0x1203F, 0x12040, 0x12041, 0x12042, 0x12043, 0x12044, 0x12045, 0x12046, 
	0x12047, 0x12048, 0x12049, 0x1204A, 0x1204B, 0x1204C, 0x1204D, 0x1204E, 
	0x1204F, 0x12050, 0x12051, 0x12052, 0x12053, 0x12054, 0x12055, 0x12056, 
	0x12057, 0x12058, 0x12059, 0x1205A, 0x1205B, 0x1205C, 0x1205D, 0x1205E, 
	0x1205F, 0x12060, 0x12061, 0x12062, 0x12063, 0x12064, 0x12065, 0x12066, 
	0x12067, 0x12068, 0x12069, 0x1206A, 0x1206B, 0x1206C, 0x1206D, 0x1206E, 
	0x1206F, 0x12070, 0x12071, 0x12072, 0x12073, 0x12074, 0x12075, 0x12076, 
	0x12077, 0x12078, 0x12079, 0x1207A, 0x1207B, 0x1207C, 0x1207D, 0x1207E, 
	0x1207F, 0x12080, 0x12081, 0x12082, 0x12083, 0x12084, 0x12085, 0x12086, 
	0x12087, 0x12088, 0x12089, 0x1208A, 0x1208B, 0x1208C, 0x1208D, 0x1208E, 
	0x1208F, 0x12090, 0x12091, 0x12092, 0x12093, 0x12094, 0x12095, 0x12096, 
	0x12097, 0x12098, 0x12099, 0x1209A, 0x1209B, 0x1209C, 0x1209D, 0x1209E, 
	0x1209F, 0x120A0, 0x120A1, 0x120A2, 0x120A3, 0x120A4, 0x120A5, 0x120A6, 
	0x120A7, 0x120A8, 0x120A9, 0x120AA, 0x120AB, 0x120AC, 0x120AD, 0x120AE, 
	0x120AF, 0x120B0, 0x120B1, 0x120B2, 0x120B3, 0x120B4, 0x120B5, 0x120B6, 
	0x120B7, 0x120B8, 0x120B9, 0x120BA, 0x120BB, 0x120BC, 0x120BD, 0x120BE, 
	0x120BF, 0x120C0, 0x120C1, 0x120C2, 0x120C3, 0x120C4, 0x120C5, 0x120C6, 
	0x120C7, 0x120C8, 0x120C9, 0x120CA, 0x120CB, 0x120CC, 0x120CD, 0x120CE, 
	0x120CF, 0x120D0, 0x120D1, 0x120D2, 0x120D3, 0x120D4, 0x120D5, 0x120D6, 
	0x120D7, 0x120D8, 0x120D9, 0x120DA, 0x120DB, 0x120DC, 0x120DD, 0x120DE, 
	0x120DF, 0x120E0, 0x120E1, 0x120E2, 0x120E3, 0x120E4, 0x120E5, 0x120E6, 
	0x120E7, 0x120E8, 0x120E9, 0x120EA, 0x120EB, 0x120EC, 0x120ED, 0x120EE, 
	0x120EF, 0x120F0, 0x120F1, 0x120F2, 0x120F3, 0x120F4, 0x120F5, 0x120F6, 
	0x120F7, 0x120F8, 0x120F9, 0x120FA, 0x120FB, 0x120FC, 0x120FD, 0x120FE, 
	0x120FF, 0x12100, 0x12101, 0x12102, 0x12103, 0x12104, 0x12105, 0x12106, 
	0x12107, 0x12108, 0x12109, 0x1210A, 0x1210B, 0x1210C, 0x1210D, 0x1210E, 
	0x1210F, 0x12110, 0x12111, 0x12112, 0x12113, 0x12114, 0x12115, 0x12116, 
	0x12117, 0x12118, 0x12119, 0x1211A, 0x1211B, 0x1211C, 0x1211D, 0x1211E, 
	0x1211F, 0x12120, 0x12121, 0x12122, 0x12123, 0x12124, 0x12125, 0x12126, 
	0x12127, 0x12128, 0x12129, 0x1212A, 0x1212B, 0x1212C, 0x1212D, 0x1212E, 
	0x1212F, 0x12130, 0x12131, 0x12132, 0x12133, 0x12134, 0x12135, 0x12136, 
	0x12137, 0x12138, 0x12139, 0x1213A, 0x1213B, 0x1213C, 0x1213D, 0x1213E, 
	0x1213F, 0x12140, 0x12141, 0x12142, 0x12143, 0x12144, 0x12145, 0x12146, 
	0x12147, 0x12148, 0x12149, 0x1214A, 0x1214B, 0x1214C, 0x1214D, 0x1214E, 
	0x1214F, 0x12150, 0x12151, 0x12152, 0x12153, 0x12154, 0x12155, 0x12156, 
	0x12157, 0x12158, 0x12159, 0x1215A, 0x1215B, 0x1215C, 0x1215D, 0x1215E, 
	0x1215F, 0x12160, 0x12161, 0x12162, 0x12163, 0x12164, 0x12165, 0x12166, 
	0x12167, 0x12168, 0x12169, 0x1216A, 0x1216B, 0x1216C, 0x1216D, 0x1216E, 
	0x1216F, 0x12170, 0x12171, 0x12172, 0x12173, 0x12174, 0x12175, 0x12176, 
	0x12177, 0x12178, 0x12179, 0x1217A, 0x1217B, 0x1217C, 0x1217D, 0x1217E, 
	0x1217F, 0x12180, 0x12181, 0x12182, 0x12183, 0x12184, 0x12185, 0x12186, 
	0x12187, 0x12188, 0x12189, 0x1218A, 0x1218B, 0x1218C, 0x1218D, 0x1218E, 
	0x1218F, 0x12190, 0x12191, 0x12192, 0x12193, 0x12194, 0x12195, 0x12196, 
	0x12197, 0x12198, 0x12199, 0x1219A, 0x1219B, 0x1219C, 0x1219D, 0x1219E, 
	0x1219F, 0x121A0, 0x121A1, 0x121A2, 0x121A3, 0x121A4, 0x121A5, 0x121A6, 
	0x121A7, 0x121A8, 0x121A9, 0x121AA, 0x121AB, 0x121AC, 0x121AD, 0x121AE, 
	0x121AF, 0x121B0, 0x121B1, 0x121B2, 0x121B3, 0x121B4, 0x121B5, 0x121B6, 
	0x121B7, 0x121B8, 0x121B9, 0x121BA, 0x121BB, 0x121BC, 0x121BD, 0x121BE, 
	0x121BF, 0x121C0, 0x121C1, 0x121C2, 0x121C3, 0x121C4, 0x121C5, 0x121C6, 
	0x121C7, 0x121C8, 0x121C9, 0x121CA, 0x121CB, 0x121CC, 0x121CD, 0x121CE, 
	0x121CF, 0x121D0, 0x121D1, 0x121D2, 0x121D3, 0x121D4, 0x121D5, 0x121D6, 
	0x121D7, 0x121D8, 0x121D9, 0x121DA, 0x121DB, 0x121DC, 0x121DD, 0x121DE, 
	0x121DF, 0x121E0, 0x121E1, 0x121E2, 0x121E3, 0x121E4, 0x121E5, 0x121E6, 
	0x121E7, 0x121E8, 0x121E9, 0x121EA, 0x121EB, 0x121EC, 0x121ED, 0x121EE, 
	0x121EF, 0x121F0, 0x121F1, 0x121F2, 0x121F3, 0x121F4, 0x121F5, 0x121F6, 
	0x121F7, 0x121F8, 0x121F9, 0x121FA, 0x121FB, 0x121FC, 0x121FD, 0x121FE, 
	0x121FF, 0x12200, 0x12201, 0x12202, 0x12203, 0x12204, 0x12205, 0x12206, 
	0x12207, 0x12208, 0x12209, 0x1220A, 0x1220B, 0x1220C, 0x1220D, 0x1220E, 
	0x1220F, 0x12210, 0x12211, 0x12212, 0x12213, 0x12214, 0x12215, 0x12216, 
	0x12217, 0x12218, 0x12219, 0x1221A, 0x1221B, 0x1221C, 0x1221D, 0x1221E, 
	0x1221F, 0x12220, 0x12221, 0x12222, 0x12223, 0x12224, 0x12225, 0x12226, 
	0x12227, 0x12228, 0x12229, 0x1222A, 0x1222B, 0x1222C, 0x1222D, 0x1222E, 
	0x1222F, 0x12230, 0x12231, 0x12232, 0x12233, 0x12234, 0x12235, 0x12236, 
	0x12237, 0x12238, 0x12239, 0x1223A, 0x1223B, 0x1223C, 0x1223D, 0x1223E, 
	0x1223F, 0x12240, 0x12241, 0x12242, 0x12243, 0x12244, 0x12245, 0x12246, 
	0x12247, 0x12248, 0x12249, 0x1224A, 0x1224B, 0x1224C, 0x1224D, 0x1224E, 
	0x1224F, 0x12250, 0x12251, 0x12252, 0x12253, 0x12254, 0x12255, 0x12256, 
	0x12257, 0x12258, 0x12259, 0x1225A, 0x1225B, 0x1225C, 0x1225D, 0x1225E, 
	0x1225F, 0x12260, 0x12261, 0x12262, 0x12263, 0x12264, 0x12265, 0x12266, 
	0x12267, 0x12268, 0x12269, 0x1226A, 0x1226B, 0x1226C, 0x1226D, 0x1226E, 
	0x1226F, 0x12270, 0x12271, 0x12272, 0x12273, 0x12274, 0x12275, 0x12276, 
	0x12277, 0x12278, 0x12279, 0x1227A, 0x1227B, 0x1227C, 0x1227D, 0x1227E, 
	0x1227F, 0x12280, 0x12281, 0x12282, 0x12283, 0x12284, 0x12285, 0x12286, 
	0x12287, 0x12288, 0x12289, 0x1228A, 0x1228B, 0x1228C, 0x1228D, 0x1228E, 
	0x1228F, 0x12290, 0x12291, 0x12292, 0x12293, 0x12294, 0x12295, 0x12296, 
	0x12297, 0x12298, 0x12299, 0x1229A, 0x1229B, 0x1229C, 0x1229D, 0x1229E, 
	0x1229F, 0x122A0, 0x122A1, 0x122A2, 0x122A3, 0x122A4, 0x122A5, 0x122A6, 
	0x122A7, 0x122A8, 0x122A9, 0x122AA, 0x122AB, 0x122AC, 0x122AD, 0x122AE, 
	0x122AF, 0x122B0, 0x122B1, 0x122B2, 0x122B3, 0x122B4, 0x122B5, 0x122B6, 
	0x122B7, 0x122B8, 0x122B9, 0x122BA, 0x122BB, 0x122BC, 0x122BD, 0x122BE, 
	0x122BF, 0x122C0, 0x122C1, 0x122C2, 0x122C3, 0x122C4, 0x122C5, 0x122C6, 
	0x122C7, 0x122C8, 0x122C9, 0x122CA, 0x122CB, 0x122CC, 0x122CD, 0x122CE, 
	0x122CF, 0x122D0, 0x122D1, 0x122D2, 0x122D3, 0x122D4, 0x122D5, 0x122D6, 
	0x122D7, 0x122D8, 0x122D9, 0x122DA, 0x122DB, 0x122DC, 0x122DD, 0x122DE, 
	0x122DF, 0x122E0, 0x122E1, 0x122E2, 0x122E3, 0x122E4, 0x122E5, 0x122E6, 
	0x122E7, 0x122E8, 0x122E9, 0x122EA, 0x122EB, 0x122EC, 0x122ED, 0x122EE, 
	0x122EF, 0x122F0, 0x122F1, 0x122F2, 0x122F3, 0x122F4, 0x122F5, 0x122F6, 
	0x122F7, 0x122F8, 0x122F9, 0x122FA, 0x122FB, 0x122FC, 0x122FD, 0x122FE, 
	0x122FF, 0x12300, 0x12301, 0x12302, 0x12303, 0x12304, 0x12305, 0x12306, 
	0x12307, 0x12308, 0x12309, 0x1230A, 0x1230B, 0x1230C, 0x1230D, 0x1230E, 
	0x1230F, 0x12310, 0x12311, 0x12312, 0x12313, 0x12314, 0x12315, 0x12316, 
	0x12317, 0x12318, 0x12319, 0x1231A, 0x1231B, 0x1231C, 0x1231D, 0x1231E, 
	0x1231F, 0x12320, 0x12321, 0x12322, 0x12323, 0x12324, 0x12325, 0x12326, 
	0x12327, 0x12328, 0x12329, 0x1232A, 0x1232B, 0x1232C, 0x1232D, 0x1232E, 
	0x1232F, 0x12330, 0x12331, 0x12332, 0x12333, 0x12334, 0x12335, 0x12336, 
	0x12337, 0x12338, 0x12339, 0x1233A, 0x1233B, 0x1233C, 0x1233D, 0x1233E, 
	0x1233F, 0x12340, 0x12341, 0x12342, 0x12343, 0x12344, 0x12345, 0x12346, 
	0x12347, 0x12348, 0x12349, 0x1234A, 0x1234B, 0x1234C, 0x1234D, 0x1234E, 
	0x1234F, 0x12350, 0x12351, 0x12352, 0x12353, 0x12354, 0x12355, 0x12356, 
	0x12357, 0x12358, 0x12359, 0x1235A, 0x1235B, 0x1235C, 0x1235D, 0x1235E, 
	0x1235F, 0x12360, 0x12361, 0x12362, 0x12363, 0x12364, 0x12365, 0x12366, 
	0x12367, 0x12368, 0x12369, 0x1236A, 0x1236B, 0x1236C, 0x1236D, 0x1236E, 
	0x12400, 0x12401, 0x12402, 0x12403, 0x12404, 0x12405, 0x12406, 0x12407, 
	0x12408, 0x12409, 0x1240A, 0x1240B, 0x1240C, 0x1240D, 0x1240E, 0x1240F, 
	0x12410, 0x12411, 0x12412, 0x12413, 0x12414, 0x12415, 0x12416, 0x12417, 
	0x12418, 0x12419, 0x1241A, 0x1241B, 0x1241C, 0x1241D, 0x1241E, 0x1241F, 
	0x12420, 0x12421, 0x12422, 0x12423, 0x12424, 0x12425, 0x12426, 0x12427, 
	0x12428, 0x12429, 0x1242A, 0x1242B, 0x1242C, 0x1242D, 0x1242E, 0x1242F, 
	0x12430, 0x12431, 0x12432, 0x12433, 0x12434, 0x12435, 0x12436, 0x12437, 
	0x12438, 0x12439, 0x1243A, 0x1243B, 0x1243C, 0x1243D, 0x1243E, 0x1243F, 
	0x12440, 0x12441, 0x12442, 0x12443, 0x12444, 0x12445, 0x12446, 0x12447, 
	0x12448, 0x12449, 0x1244A, 0x1244B, 0x1244C, 0x1244D, 0x1244E, 0x1244F, 
	0x12450, 0x12451, 0x12452, 0x12453, 0x12454, 0x12455, 0x12456, 0x12457, 
	0x12458, 0x12459, 0x1245A, 0x1245B, 0x1245C, 0x1245D, 0x1245E, 0x1245F, 
	0x12460, 0x12461, 0x12462, 0x12470, 0x12471, 0x12472, 0x12473, 0x1D000, 
	0x1D001, 0x1D002, 0x1D003, 0x1D004, 0x1D005, 0x1D006, 0x1D007, 0x1D008, 
	0x1D009, 0x1D00A, 0x1D00B, 0x1D00C, 0x1D00D, 0x1D00E, 0x1D00F, 0x1D010, 
	0x1D011, 0x1D012, 0x1D013, 0x1D014, 0x1D015, 0x1D016, 0x1D017, 0x1D018, 
	0x1D019, 0x1D01A, 0x1D01B, 0x1D01C, 0x1D01D, 0x1D01E, 0x1D01F, 0x1D020, 
	0x1D021, 0x1D022, 0x1D023, 0x1D024, 0x1D025, 0x1D026, 0x1D027, 0x1D028, 
	0x1D029, 0x1D02A, 0x1D02B, 0x1D02C, 0x1D02D, 0x1D02E, 0x1D02F, 0x1D030, 
	0x1D031, 0x1D032, 0x1D033, 0x1D034, 0x1D035, 0x1D036, 0x1D037, 0x1D038, 
	0x1D039, 0x1D03A, 0x1D03B, 0x1D03C, 0x1D03D, 0x1D03E, 0x1D03F, 0x1D040, 
	0x1D041, 0x1D042, 0x1D043, 0x1D044, 0x1D045, 0x1D046, 0x1D047, 0x1D048, 
	0x1D049, 0x1D04A, 0x1D04B, 0x1D04C, 0x1D04D, 0x1D04E, 0x1D04F, 0x1D050, 
	0x1D051, 0x1D052, 0x1D053, 0x1D054, 0x1D055, 0x1D056, 0x1D057, 0x1D058, 
	0x1D059, 0x1D05A, 0x1D05B, 0x1D05C, 0x1D05D, 0x1D05E, 0x1D05F, 0x1D060, 
	0x1D061, 0x1D062, 0x1D063, 0x1D064, 0x1D065, 0x1D066, 0x1D067, 0x1D068, 
	0x1D069, 0x1D06A, 0x1D06B, 0x1D06C, 0x1D06D, 0x1D06E, 0x1D06F, 0x1D070, 
	0x1D071, 0x1D072, 0x1D073, 0x1D074, 0x1D075, 0x1D076, 0x1D077, 0x1D078, 
	0x1D079, 0x1D07A, 0x1D07B, 0x1D07C, 0x1D07D, 0x1D07E, 0x1D07F, 0x1D080, 
	0x1D081, 0x1D082, 0x1D083, 0x1D084, 0x1D085, 0x1D086, 0x1D087, 0x1D088, 
	0x1D089, 0x1D08A, 0x1D08B, 0x1D08C, 0x1D08D, 0x1D08E, 0x1D08F, 0x1D090, 
	0x1D091, 0x1D092, 0x1D093, 0x1D094, 0x1D095, 0x1D096, 0x1D097, 0x1D098, 
	0x1D099, 0x1D09A, 0x1D09B, 0x1D09C, 0x1D09D, 0x1D09E, 0x1D09F, 0x1D0A0, 
	0x1D0A1, 0x1D0A2, 0x1D0A3, 0x1D0A4, 0x1D0A5, 0x1D0A6, 0x1D0A7, 0x1D0A8, 
	0x1D0A9, 0x1D0AA, 0x1D0AB, 0x1D0AC, 0x1D0AD, 0x1D0AE, 0x1D0AF, 0x1D0B0, 
	0x1D0B1, 0x1D0B2, 0x1D0B3, 0x1D0B4, 0x1D0B5, 0x1D0B6, 0x1D0B7, 0x1D0B8, 
	0x1D0B9, 0x1D0BA, 0x1D0BB, 0x1D0BC, 0x1D0BD, 0x1D0BE, 0x1D0BF, 0x1D0C0, 
	0x1D0C1, 0x1D0C2, 0x1D0C3, 0x1D0C4, 0x1D0C5, 0x1D0C6, 0x1D0C7, 0x1D0C8, 
	0x1D0C9, 0x1D0CA, 0x1D0CB, 0x1D0CC, 0x1D0CD, 0x1D0CE, 0x1D0CF, 0x1D0D0, 
	0x1D0D1, 0x1D0D2, 0x1D0D3, 0x1D0D4, 0x1D0D5, 0x1D0D6, 0x1D0D7, 0x1D0D8, 
	0x1D0D9, 0x1D0DA, 0x1D0DB, 0x1D0DC, 0x1D0DD, 0x1D0DE, 0x1D0DF, 0x1D0E0, 
	0x1D0E1, 0x1D0E2, 0x1D0E3, 0x1D0E4, 0x1D0E5, 0x1D0E6, 0x1D0E7, 0x1D0E8, 
	0x1D0E9, 0x1D0EA, 0x1D0EB, 0x1D0EC, 0x1D0ED, 0x1D0EE, 0x1D0EF, 0x1D0F0, 
	0x1D0F1, 0x1D0F2, 0x1D0F3, 0x1D0F4, 0x1D0F5, 0x1D100, 0x1D101, 0x1D102, 
	0x1D103, 0x1D104, 0x1D105, 0x1D106, 0x1D107, 0x1D108, 0x1D109, 0x1D10A, 
	0x1D10B, 0x1D10C, 0x1D10D, 0x1D10E, 0x1D10F, 0x1D110, 0x1D111, 0x1D112, 
	0x1D113, 0x1D114, 0x1D115, 0x1D116, 0x1D117, 0x1D118, 0x1D119, 0x1D11A, 
	0x1D11B, 0x1D11C, 0x1D11D, 0x1D11E, 0x1D11F, 0x1D120, 0x1D121, 0x1D122, 
	0x1D123, 0x1D124, 0x1D125, 0x1D126, 0x1D12A, 0x1D12B, 0x1D12C, 0x1D12D, 
	0x1D12E, 0x1D12F, 0x1D130, 0x1D131, 0x1D132, 0x1D133, 0x1D134, 0x1D135, 
	0x1D136, 0x1D137, 0x1D138, 0x1D139, 0x1D13A, 0x1D13B, 0x1D13C, 0x1D13D, 
	0x1D13E, 0x1D13F, 0x1D140, 0x1D141, 0x1D142, 0x1D143, 0x1D144, 0x1D145, 
	0x1D146, 0x1D147, 0x1D148, 0x1D149, 0x1D14A, 0x1D14B, 0x1D14C, 0x1D14D, 
	0x1D14E, 0x1D14F, 0x1D150, 0x1D151, 0x1D152, 0x1D153, 0x1D154, 0x1D155, 
	0x1D156, 0x1D157, 0x1D158, 0x1D159, 0x1D15A, 0x1D15B, 0x1D15C, 0x1D15D, 
	0x1D15E, 0x1D15F, 0x1D160, 0x1D161, 0x1D162, 0x1D163, 0x1D164, 0x1D165, 
	0x1D166, 0x1D167, 0x1D168, 0x1D169, 0x1D16A, 0x1D16B, 0x1D16C, 0x1D16D, 
	0x1D16E, 0x1D16F, 0x1D170, 0x1D171, 0x1D172, 0x1D173, 0x1D174, 0x1D175, 
	0x1D176, 0x1D177, 0x1D178, 0x1D179, 0x1D17A, 0x1D17B, 0x1D17C, 0x1D17D, 
	0x1D17E, 0x1D17F, 0x1D180, 0x1D181, 0x1D182, 0x1D183, 0x1D184, 0x1D185, 
	0x1D186, 0x1D187, 0x1D188, 0x1D189, 0x1D18A, 0x1D18B, 0x1D18C, 0x1D18D, 
	0x1D18E, 0x1D18F, 0x1D190, 0x1D191, 0x1D192, 0x1D193, 0x1D194, 0x1D195, 
	0x1D196, 0x1D197, 0x1D198, 0x1D199, 0x1D19A, 0x1D19B, 0x1D19C, 0x1D19D, 
	0x1D19E, 0x1D19F, 0x1D1A0, 0x1D1A1, 0x1D1A2, 0x1D1A3, 0x1D1A4, 0x1D1A5, 
	0x1D1A6, 0x1D1A7, 0x1D1A8, 0x1D1A9, 0x1D1AA, 0x1D1AB, 0x1D1AC, 0x1D1AD, 
	0x1D1AE, 0x1D1AF, 0x1D1B0, 0x1D1B1, 0x1D1B2, 0x1D1B3, 0x1D1B4, 0x1D1B5, 
	0x1D1B6, 0x1D1B7, 0x1D1B8, 0x1D1B9, 0x1D1BA, 0x1D1BB, 0x1D1BC, 0x1D1BD, 
	0x1D1BE, 0x1D1BF, 0x1D1C0, 0x1D1C1, 0x1D1C2, 0x1D1C3, 0x1D1C4, 0x1D1C5, 
	0x1D1C6, 0x1D1C7, 0x1D1C8, 0x1D1C9, 0x1D1CA, 0x1D1CB, 0x1D1CC, 0x1D1CD, 
	0x1D1CE, 0x1D1CF, 0x1D1D0, 0x1D1D1, 0x1D1D2, 0x1D1D3, 0x1D1D4, 0x1D1D5, 
	0x1D1D6, 0x1D1D7, 0x1D1D8, 0x1D1D9, 0x1D1DA, 0x1D1DB, 0x1D1DC, 0x1D1DD, 
	0x1D200, 0x1D201, 0x1D202, 0x1D203, 0x1D204, 0x1D205, 0x1D206, 0x1D207, 
	0x1D208, 0x1D209, 0x1D20A, 0x1D20B, 0x1D20C, 0x1D20D, 0x1D20E, 0x1D20F, 
	0x1D210, 0x1D211, 0x1D212, 0x1D213, 0x1D214, 0x1D215, 0x1D216, 0x1D217, 
	0x1D218, 0x1D219, 0x1D21A, 0x1D21B, 0x1D21C, 0x1D21D, 0x1D21E, 0x1D21F, 
	0x1D220, 0x1D221, 0x1D222, 0x1D223, 0x1D224, 0x1D225, 0x1D226, 0x1D227, 
	0x1D228, 0x1D229, 0x1D22A, 0x1D22B, 0x1D22C, 0x1D22D, 0x1D22E, 0x1D22F, 
	0x1D230, 0x1D231, 0x1D232, 0x1D233, 0x1D234, 0x1D235, 0x1D236, 0x1D237, 
	0x1D238, 0x1D239, 0x1D23A, 0x1D23B, 0x1D23C, 0x1D23D, 0x1D23E, 0x1D23F, 
	0x1D240, 0x1D241, 0x1D242, 0x1D243, 0x1D244, 0x1D245, 0x1D300, 0x1D301, 
	0x1D302, 0x1D303, 0x1D304, 0x1D305, 0x1D306, 0x1D307, 0x1D308, 0x1D309, 
	0x1D30A, 0x1D30B, 0x1D30C, 0x1D30D, 0x1D30E, 0x1D30F, 0x1D310, 0x1D311, 
	0x1D312, 0x1D313, 0x1D314, 0x1D315, 0x1D316, 0x1D317, 0x1D318, 0x1D319, 
	0x1D31A, 0x1D31B, 0x1D31C, 0x1D31D, 0x1D31E, 0x1D31F, 0x1D320, 0x1D321, 
	0x1D322, 0x1D323, 0x1D324, 0x1D325, 0x1D326, 0x1D327, 0x1D328, 0x1D329, 
	0x1D32A, 0x1D32B, 0x1D32C, 0x1D32D, 0x1D32E, 0x1D32F, 0x1D330, 0x1D331, 
	0x1D332, 0x1D333, 0x1D334, 0x1D335, 0x1D336, 0x1D337, 0x1D338, 0x1D339, 
	0x1D33A, 0x1D33B, 0x1D33C, 0x1D33D, 0x1D33E, 0x1D33F, 0x1D340, 0x1D341, 
	0x1D342, 0x1D343, 0x1D344, 0x1D345, 0x1D346, 0x1D347, 0x1D348, 0x1D349, 
	0x1D34A, 0x1D34B, 0x1D34C, 0x1D34D, 0x1D34E, 0x1D34F, 0x1D350, 0x1D351, 
	0x1D352, 0x1D353, 0x1D354, 0x1D355, 0x1D356, 0x1D360, 0x1D361, 0x1D362, 
	0x1D363, 0x1D364, 0x1D365, 0x1D366, 0x1D367, 0x1D368, 0x1D369, 0x1D36A, 
	0x1D36B, 0x1D36C, 0x1D36D, 0x1D36E, 0x1D36F, 0x1D370, 0x1D371, 0x1D400, 
	0x1D401, 0x1D402, 0x1D403, 0x1D404, 0x1D405, 0x1D406, 0x1D407, 0x1D408, 
	0x1D409, 0x1D40A, 0x1D40B, 0x1D40C, 0x1D40D, 0x1D40E, 0x1D40F, 0x1D410, 
	0x1D411, 0x1D412, 0x1D413, 0x1D414, 0x1D415, 0x1D416, 0x1D417, 0x1D418, 
	0x1D419, 0x1D41A, 0x1D41B, 0x1D41C, 0x1D41D, 0x1D41E, 0x1D41F, 0x1D420, 
	0x1D421, 0x1D422, 0x1D423, 0x1D424, 0x1D425, 0x1D426, 0x1D427, 0x1D428, 
	0x1D429, 0x1D42A, 0x1D42B, 0x1D42C, 0x1D42D, 0x1D42E, 0x1D42F, 0x1D430, 
	0x1D431, 0x1D432, 0x1D433, 0x1D434, 0x1D435, 0x1D436, 0x1D437, 0x1D438, 
	0x1D439, 0x1D43A, 0x1D43B, 0x1D43C, 0x1D43D, 0x1D43E, 0x1D43F, 0x1D440, 
	0x1D441, 0x1D442, 0x1D443, 0x1D444, 0x1D445, 0x1D446, 0x1D447, 0x1D448, 
	0x1D449, 0x1D44A, 0x1D44B, 0x1D44C, 0x1D44D, 0x1D44E, 0x1D44F, 0x1D450, 
	0x1D451, 0x1D452, 0x1D453, 0x1D454, 0x1D456, 0x1D457, 0x1D458, 0x1D459, 
	0x1D45A, 0x1D45B, 0x1D45C, 0x1D45D, 0x1D45E, 0x1D45F, 0x1D460, 0x1D461, 
	0x1D462, 0x1D463, 0x1D464, 0x1D465, 0x1D466, 0x1D467, 0x1D468, 0x1D469, 
	0x1D46A, 0x1D46B, 0x1D46C, 0x1D46D, 0x1D46E, 0x1D46F, 0x1D470, 0x1D471, 
	0x1D472, 0x1D473, 0x1D474, 0x1D475, 0x1D476, 0x1D477, 0x1D478, 0x1D479, 
	0x1D47A, 0x1D47B, 0x1D47C, 0x1D47D, 0x1D47E, 0x1D47F, 0x1D480, 0x1D481, 
	0x1D482, 0x1D483, 0x1D484, 0x1D485, 0x1D486, 0x1D487, 0x1D488, 0x1D489, 
	0x1D48A, 0x1D48B, 0x1D48C, 0x1D48D, 0x1D48E, 0x1D48F, 0x1D490, 0x1D491, 
	0x1D492, 0x1D493, 0x1D494, 0x1D495, 0x1D496, 0x1D497, 0x1D498, 0x1D499, 
	0x1D49A, 0x1D49B, 0x1D49C, 0x1D49E, 0x1D49F, 0x1D4A2, 0x1D4A5, 0x1D4A6, 
	0x1D4A9, 0x1D4AA, 0x1D4AB, 0x1D4AC, 0x1D4AE, 0x1D4AF, 0x1D4B0, 0x1D4B1, 
	0x1D4B2, 0x1D4B3, 0x1D4B4, 0x1D4B5, 0x1D4B6, 0x1D4B7, 0x1D4B8, 0x1D4B9, 
	0x1D4BB, 0x1D4BD, 0x1D4BE, 0x1D4BF, 0x1D4C0, 0x1D4C1, 0x1D4C2, 0x1D4C3, 
	0x1D4C5, 0x1D4C6, 0x1D4C7, 0x1D4C8, 0x1D4C9, 0x1D4CA, 0x1D4CB, 0x1D4CC, 
	0x1D4CD, 0x1D4CE, 0x1D4CF, 0x1D4D0, 0x1D4D1, 0x1D4D2, 0x1D4D3, 0x1D4D4, 
	0x1D4D5, 0x1D4D6, 0x1D4D7, 0x1D4D8, 0x1D4D9, 0x1D4DA, 0x1D4DB, 0x1D4DC, 
	0x1D4DD, 0x1D4DE, 0x1D4DF, 0x1D4E0, 0x1D4E1, 0x1D4E2, 0x1D4E3, 0x1D4E4, 
	0x1D4E5, 0x1D4E6, 0x1D4E7, 0x1D4E8, 0x1D4E9, 0x1D4EA, 0x1D4EB, 0x1D4EC, 
	0x1D4ED, 0x1D4EE, 0x1D4EF, 0x1D4F0, 0x1D4F1, 0x1D4F2, 0x1D4F3, 0x1D4F4, 
	0x1D4F5, 0x1D4F6, 0x1D4F7, 0x1D4F8, 0x1D4F9, 0x1D4FA, 0x1D4FB, 0x1D4FC, 
	0x1D4FD, 0x1D4FE, 0x1D4FF, 0x1D500, 0x1D501, 0x1D502, 0x1D503, 0x1D504, 
	0x1D505, 0x1D507, 0x1D508, 0x1D509, 0x1D50A, 0x1D50D, 0x1D50E, 0x1D50F, 
	0x1D510, 0x1D511, 0x1D512, 0x1D513, 0x1D514, 0x1D516, 0x1D517, 0x1D518, 
	0x1D519, 0x1D51A, 0x1D51B, 0x1D51C, 0x1D51E, 0x1D51F, 0x1D520, 0x1D521, 
	0x1D522, 0x1D523, 0x1D524, 0x1D525, 0x1D526, 0x1D527, 0x1D528, 0x1D529, 
	0x1D52A, 0x1D52B, 0x1D52C, 0x1D52D, 0x1D52E, 0x1D52F, 0x1D530, 0x1D531, 
	0x1D532, 0x1D533, 0x1D534, 0x1D535, 0x1D536, 0x1D537, 0x1D538, 0x1D539, 
	0x1D53B, 0x1D53C, 0x1D53D, 0x1D53E, 0x1D540, 0x1D541, 0x1D542, 0x1D543, 
	0x1D544, 0x1D546, 0x1D54A, 0x1D54B, 0x1D54C, 0x1D54D, 0x1D54E, 0x1D54F, 
	0x1D550, 0x1D552, 0x1D553, 0x1D554, 0x1D555, 0x1D556, 0x1D557, 0x1D558, 
	0x1D559, 0x1D55A, 0x1D55B, 0x1D55C, 0x1D55D, 0x1D55E, 0x1D55F, 0x1D560, 
	0x1D561, 0x1D562, 0x1D563, 0x1D564, 0x1D565, 0x1D566, 0x1D567, 0x1D568, 
	0x1D569, 0x1D56A, 0x1D56B, 0x1D56C, 0x1D56D, 0x1D56E, 0x1D56F, 0x1D570, 
	0x1D571, 0x1D572, 0x1D573, 0x1D574, 0x1D575, 0x1D576, 0x1D577, 0x1D578, 
	0x1D579, 0x1D57A, 0x1D57B, 0x1D57C, 0x1D57D, 0x1D57E, 0x1D57F, 0x1D580, 
	0x1D581, 0x1D582, 0x1D583, 0x1D584, 0x1D585, 0x1D586, 0x1D587, 0x1D588, 
	0x1D589, 0x1D58A, 0x1D58B, 0x1D58C, 0x1D58D, 0x1D58E, 0x1D58F, 0x1D590, 
	0x1D591, 0x1D592, 0x1D593, 0x1D594, 0x1D595, 0x1D596, 0x1D597, 0x1D598, 
	0x1D599, 0x1D59A, 0x1D59B, 0x1D59C, 0x1D59D, 0x1D59E, 0x1D59F, 0x1D5A0, 
	0x1D5A1, 0x1D5A2, 0x1D5A3, 0x1D5A4, 0x1D5A5, 0x1D5A6, 0x1D5A7, 0x1D5A8, 
	0x1D5A9, 0x1D5AA, 0x1D5AB, 0x1D5AC, 0x1D5AD, 0x1D5AE, 0x1D5AF, 0x1D5B0, 
	0x1D5B1, 0x1D5B2, 0x1D5B3, 0x1D5B4, 0x1D5B5, 0x1D5B6, 0x1D5B7, 0x1D5B8, 
	0x1D5B9, 0x1D5BA, 0x1D5BB, 0x1D5BC, 0x1D5BD, 0x1D5BE, 0x1D5BF, 0x1D5C0, 
	0x1D5C1, 0x1D5C2, 0x1D5C3, 0x1D5C4, 0x1D5C5, 0x1D5C6, 0x1D5C7, 0x1D5C8, 
	0x1D5C9, 0x1D5CA, 0x1D5CB, 0x1D5CC, 0x1D5CD, 0x1D5CE, 0x1D5CF, 0x1D5D0, 
	0x1D5D1, 0x1D5D2, 0x1D5D3, 0x1D5D4, 0x1D5D5, 0x1D5D6, 0x1D5D7, 0x1D5D8, 
	0x1D5D9, 0x1D5DA, 0x1D5DB, 0x1D5DC, 0x1D5DD, 0x1D5DE, 0x1D5DF, 0x1D5E0, 
	0x1D5E1, 0x1D5E2, 0x1D5E3, 0x1D5E4, 0x1D5E5, 0x1D5E6, 0x1D5E7, 0x1D5E8, 
	0x1D5E9, 0x1D5EA, 0x1D5EB, 0x1D5EC, 0x1D5ED, 0x1D5EE, 0x1D5EF, 0x1D5F0, 
	0x1D5F1, 0x1D5F2, 0x1D5F3, 0x1D5F4, 0x1D5F5, 0x1D5F6, 0x1D5F7, 0x1D5F8, 
	0x1D5F9, 0x1D5FA, 0x1D5FB, 0x1D5FC, 0x1D5FD, 0x1D5FE, 0x1D5FF, 0x1D600, 
	0x1D601, 0x1D602, 0x1D603, 0x1D604, 0x1D605, 0x1D606, 0x1D607, 0x1D608, 
	0x1D609, 0x1D60A, 0x1D60B, 0x1D60C, 0x1D60D, 0x1D60E, 0x1D60F, 0x1D610, 
	0x1D611, 0x1D612, 0x1D613, 0x1D614, 0x1D615, 0x1D616, 0x1D617, 0x1D618, 
	0x1D619, 0x1D61A, 0x1D61B, 0x1D61C, 0x1D61D, 0x1D61E, 0x1D61F, 0x1D620, 
	0x1D621, 0x1D622, 0x1D623, 0x1D624, 0x1D625, 0x1D626, 0x1D627, 0x1D628, 
	0x1D629, 0x1D62A, 0x1D62B, 0x1D62C, 0x1D62D, 0x1D62E, 0x1D62F, 0x1D630, 
	0x1D631, 0x1D632, 0x1D633, 0x1D634, 0x1D635, 0x1D636, 0x1D637, 0x1D638, 
	0x1D639, 0x1D63A, 0x1D63B, 0x1D63C, 0x1D63D, 0x1D63E, 0x1D63F, 0x1D640, 
	0x1D641, 0x1D642, 0x1D643, 0x1D644, 0x1D645, 0x1D646, 0x1D647, 0x1D648, 
	0x1D649, 0x1D64A, 0x1D64B, 0x1D64C, 0x1D64D, 0x1D64E, 0x1D64F, 0x1D650, 
	0x1D651, 0x1D652, 0x1D653, 0x1D654, 0x1D655, 0x1D656, 0x1D657, 0x1D658, 
	0x1D659, 0x1D65A, 0x1D65B, 0x1D65C, 0x1D65D, 0x1D65E, 0x1D65F, 0x1D660, 
	0x1D661, 0x1D662, 0x1D663, 0x1D664, 0x1D665, 0x1D666, 0x1D667, 0x1D668, 
	0x1D669, 0x1D66A, 0x1D66B, 0x1D66C, 0x1D66D, 0x1D66E, 0x1D66F, 0x1D670, 
	0x1D671, 0x1D672, 0x1D673, 0x1D674, 0x1D675, 0x1D676, 0x1D677, 0x1D678, 
	0x1D679, 0x1D67A, 0x1D67B, 0x1D67C, 0x1D67D, 0x1D67E, 0x1D67F, 0x1D680, 
	0x1D681, 0x1D682, 0x1D683, 0x1D684, 0x1D685, 0x1D686, 0x1D687, 0x1D688, 
	0x1D689, 0x1D68A, 0x1D68B, 0x1D68C, 0x1D68D, 0x1D68E, 0x1D68F, 0x1D690, 
	0x1D691, 0x1D692, 0x1D693, 0x1D694, 0x1D695, 0x1D696, 0x1D697, 0x1D698, 
	0x1D699, 0x1D69A, 0x1D69B, 0x1D69C, 0x1D69D, 0x1D69E, 0x1D69F, 0x1D6A0, 
	0x1D6A1, 0x1D6A2, 0x1D6A3, 0x1D6A4, 0x1D6A5, 0x1D6A8, 0x1D6A9, 0x1D6AA, 
	0x1D6AB, 0x1D6AC, 0x1D6AD, 0x1D6AE, 0x1D6AF, 0x1D6B0, 0x1D6B1, 0x1D6B2, 
	0x1D6B3, 0x1D6B4, 0x1D6B5, 0x1D6B6, 0x1D6B7, 0x1D6B8, 0x1D6B9, 0x1D6BA, 
	0x1D6BB, 0x1D6BC, 0x1D6BD, 0x1D6BE, 0x1D6BF, 0x1D6C0, 0x1D6C1, 0x1D6C2, 
	0x1D6C3, 0x1D6C4, 0x1D6C5, 0x1D6C6, 0x1D6C7, 0x1D6C8, 0x1D6C9, 0x1D6CA, 
	0x1D6CB, 0x1D6CC, 0x1D6CD, 0x1D6CE, 0x1D6CF, 0x1D6D0, 0x1D6D1, 0x1D6D2, 
	0x1D6D3, 0x1D6D4, 0x1D6D5, 0x1D6D6, 0x1D6D7, 0x1D6D8, 0x1D6D9, 0x1D6DA, 
	0x1D6DB, 0x1D6DC, 0x1D6DD, 0x1D6DE, 0x1D6DF, 0x1D6E0, 0x1D6E1, 0x1D6E2, 
	0x1D6E3, 0x1D6E4, 0x1D6E5, 0x1D6E6, 0x1D6E7, 0x1D6E8, 0x1D6E9, 0x1D6EA, 
	0x1D6EB, 0x1D6EC, 0x1D6ED, 0x1D6EE, 0x1D6EF, 0x1D6F0, 0x1D6F1, 0x1D6F2, 
	0x1D6F3, 0x1D6F4, 0x1D6F5, 0x1D6F6, 0x1D6F7, 0x1D6F8, 0x1D6F9, 0x1D6FA, 
	0x1D6FB, 0x1D6FC, 0x1D6FD, 0x1D6FE, 0x1D6FF, 0x1D700, 0x1D701, 0x1D702, 
	0x1D703, 0x1D704, 0x1D705, 0x1D706, 0x1D707, 0x1D708, 0x1D709, 0x1D70A, 
	0x1D70B, 0x1D70C, 0x1D70D, 0x1D70E, 0x1D70F, 0x1D710, 0x1D711, 0x1D712, 
	0x1D713, 0x1D714, 0x1D715, 0x1D716, 0x1D717, 0x1D718, 0x1D719, 0x1D71A, 
	0x1D71B, 0x1D71C, 0x1D71D, 0x1D71E, 0x1D71F, 0x1D720, 0x1D721, 0x1D722, 
	0x1D723, 0x1D724, 0x1D725, 0x1D726, 0x1D727, 0x1D728, 0x1D729, 0x1D72A, 
	0x1D72B, 0x1D72C, 0x1D72D, 0x1D72E, 0x1D72F, 0x1D730, 0x1D731, 0x1D732, 
	0x1D733, 0x1D734, 0x1D735, 0x1D736, 0x1D737, 0x1D738, 0x1D739, 0x1D73A, 
	0x1D73B, 0x1D73C, 0x1D73D, 0x1D73E, 0x1D73F, 0x1D740, 0x1D741, 0x1D742, 
	0x1D743, 0x1D744, 0x1D745, 0x1D746, 0x1D747, 0x1D748, 0x1D749, 0x1D74A, 
	0x1D74B, 0x1D74C, 0x1D74D, 0x1D74E, 0x1D74F, 0x1D750, 0x1D751, 0x1D752, 
	0x1D753, 0x1D754, 0x1D755, 0x1D756, 0x1D757, 0x1D758, 0x1D759, 0x1D75A, 
	0x1D75B, 0x1D75C, 0x1D75D, 0x1D75E, 0x1D75F, 0x1D760, 0x1D761, 0x1D762, 
	0x1D763, 0x1D764, 0x1D765, 0x1D766, 0x1D767, 0x1D768, 0x1D769, 0x1D76A, 
	0x1D76B, 0x1D76C, 0x1D76D, 0x1D76E, 0x1D76F, 0x1D770, 0x1D771, 0x1D772, 
	0x1D773, 0x1D774, 0x1D775, 0x1D776, 0x1D777, 0x1D778, 0x1D779, 0x1D77A, 
	0x1D77B, 0x1D77C, 0x1D77D, 0x1D77E, 0x1D77F, 0x1D780, 0x1D781, 0x1D782, 
	0x1D783, 0x1D784, 0x1D785, 0x1D786, 0x1D787, 0x1D788, 0x1D789, 0x1D78A, 
	0x1D78B, 0x1D78C, 0x1D78D, 0x1D78E, 0x1D78F, 0x1D790, 0x1D791, 0x1D792, 
	0x1D793, 0x1D794, 0x1D795, 0x1D796, 0x1D797, 0x1D798, 0x1D799, 0x1D79A, 
	0x1D79B, 0x1D79C, 0x1D79D, 0x1D79E, 0x1D79F, 0x1D7A0, 0x1D7A1, 0x1D7A2, 
	0x1D7A3, 0x1D7A4, 0x1D7A5, 0x1D7A6, 0x1D7A7, 0x1D7A8, 0x1D7A9, 0x1D7AA, 
	0x1D7AB, 0x1D7AC, 0x1D7AD, 0x1D7AE, 0x1D7AF, 0x1D7B0, 0x1D7B1, 0x1D7B2, 
	0x1D7B3, 0x1D7B4, 0x1D7B5, 0x1D7B6, 0x1D7B7, 0x1D7B8, 0x1D7B9, 0x1D7BA, 
	0x1D7BB, 0x1D7BC, 0x1D7BD, 0x1D7BE, 0x1D7BF, 0x1D7C0, 0x1D7C1, 0x1D7C2, 
	0x1D7C3, 0x1D7C4, 0x1D7C5, 0x1D7C6, 0x1D7C7, 0x1D7C8, 0x1D7C9, 0x1D7CA, 
	0x1D7CB, 0x1D7CE, 0x1D7CF, 0x1D7D0, 0x1D7D1, 0x1D7D2, 0x1D7D3, 0x1D7D4, 
	0x1D7D5, 0x1D7D6, 0x1D7D7, 0x1D7D8, 0x1D7D9, 0x1D7DA, 0x1D7DB, 0x1D7DC, 
	0x1D7DD, 0x1D7DE, 0x1D7DF, 0x1D7E0, 0x1D7E1, 0x1D7E2, 0x1D7E3, 0x1D7E4, 
	0x1D7E5, 0x1D7E6, 0x1D7E7, 0x1D7E8, 0x1D7E9, 0x1D7EA, 0x1D7EB, 0x1D7EC, 
	0x1D7ED, 0x1D7EE, 0x1D7EF, 0x1D7F0, 0x1D7F1, 0x1D7F2, 0x1D7F3, 0x1D7F4, 
	0x1D7F5, 0x1D7F6, 0x1D7F7, 0x1D7F8, 0x1D7F9, 0x1D7FA, 0x1D7FB, 0x1D7FC, 
	0x1D7FD, 0x1D7FE, 0x1D7FF, 0x20000, 0x2A6D6, 0x2F800, 0x2F801, 0x2F802, 
	0x2F803, 0x2F804, 0x2F805, 0x2F806, 0x2F807, 0x2F808, 0x2F809, 0x2F80A, 
	0x2F80B, 0x2F80C, 0x2F80D, 0x2F80E, 0x2F80F, 0x2F810, 0x2F811, 0x2F812, 
	0x2F813, 0x2F814, 0x2F815, 0x2F816, 0x2F817, 0x2F818, 0x2F819, 0x2F81A, 
	0x2F81B, 0x2F81C, 0x2F81D, 0x2F81E, 0x2F81F, 0x2F820, 0x2F821, 0x2F822, 
	0x2F823, 0x2F824, 0x2F825, 0x2F826, 0x2F827, 0x2F828, 0x2F829, 0x2F82A, 
	0x2F82B, 0x2F82C, 0x2F82D, 0x2F82E, 0x2F82F, 0x2F830, 0x2F831, 0x2F832, 
	0x2F833, 0x2F834, 0x2F835, 0x2F836, 0x2F837, 0x2F838, 0x2F839, 0x2F83A, 
	0x2F83B, 0x2F83C, 0x2F83D, 0x2F83E, 0x2F83F, 0x2F840, 0x2F841, 0x2F842, 
	0x2F843, 0x2F844, 0x2F845, 0x2F846, 0x2F847, 0x2F848, 0x2F849, 0x2F84A, 
	0x2F84B, 0x2F84C, 0x2F84D, 0x2F84E, 0x2F84F, 0x2F850, 0x2F851, 0x2F852, 
	0x2F853, 0x2F854, 0x2F855, 0x2F856, 0x2F857, 0x2F858, 0x2F859, 0x2F85A, 
	0x2F85B, 0x2F85C, 0x2F85D, 0x2F85E, 0x2F85F, 0x2F860, 0x2F861, 0x2F862, 
	0x2F863, 0x2F864, 0x2F865, 0x2F866, 0x2F867, 0x2F868, 0x2F869, 0x2F86A, 
	0x2F86B, 0x2F86C, 0x2F86D, 0x2F86E, 0x2F86F, 0x2F870, 0x2F871, 0x2F872, 
	0x2F873, 0x2F874, 0x2F875, 0x2F876, 0x2F877, 0x2F878, 0x2F879, 0x2F87A, 
	0x2F87B, 0x2F87C, 0x2F87D, 0x2F87E, 0x2F87F, 0x2F880, 0x2F881, 0x2F882, 
	0x2F883, 0x2F884, 0x2F885, 0x2F886, 0x2F887, 0x2F888, 0x2F889, 0x2F88A, 
	0x2F88B, 0x2F88C, 0x2F88D, 0x2F88E, 0x2F88F, 0x2F890, 0x2F891, 0x2F892, 
	0x2F893, 0x2F894, 0x2F895, 0x2F896, 0x2F897, 0x2F898, 0x2F899, 0x2F89A, 
	0x2F89B, 0x2F89C, 0x2F89D, 0x2F89E, 0x2F89F, 0x2F8A0, 0x2F8A1, 0x2F8A2, 
	0x2F8A3, 0x2F8A4, 0x2F8A5, 0x2F8A6, 0x2F8A7, 0x2F8A8, 0x2F8A9, 0x2F8AA, 
	0x2F8AB, 0x2F8AC, 0x2F8AD, 0x2F8AE, 0x2F8AF, 0x2F8B0, 0x2F8B1, 0x2F8B2, 
	0x2F8B3, 0x2F8B4, 0x2F8B5, 0x2F8B6, 0x2F8B7, 0x2F8B8, 0x2F8B9, 0x2F8BA, 
	0x2F8BB, 0x2F8BC, 0x2F8BD, 0x2F8BE, 0x2F8BF, 0x2F8C0, 0x2F8C1, 0x2F8C2, 
	0x2F8C3, 0x2F8C4, 0x2F8C5, 0x2F8C6, 0x2F8C7, 0x2F8C8, 0x2F8C9, 0x2F8CA, 
	0x2F8CB, 0x2F8CC, 0x2F8CD, 0x2F8CE, 0x2F8CF, 0x2F8D0, 0x2F8D1, 0x2F8D2, 
	0x2F8D3, 0x2F8D4, 0x2F8D5, 0x2F8D6, 0x2F8D7, 0x2F8D8, 0x2F8D9, 0x2F8DA, 
	0x2F8DB, 0x2F8DC, 0x2F8DD, 0x2F8DE, 0x2F8DF, 0x2F8E0, 0x2F8E1, 0x2F8E2, 
	0x2F8E3, 0x2F8E4, 0x2F8E5, 0x2F8E6, 0x2F8E7, 0x2F8E8, 0x2F8E9, 0x2F8EA, 
	0x2F8EB, 0x2F8EC, 0x2F8ED, 0x2F8EE, 0x2F8EF, 0x2F8F0, 0x2F8F1, 0x2F8F2, 
	0x2F8F3, 0x2F8F4, 0x2F8F5, 0x2F8F6, 0x2F8F7, 0x2F8F8, 0x2F8F9, 0x2F8FA, 
	0x2F8FB, 0x2F8FC, 0x2F8FD, 0x2F8FE, 0x2F8FF, 0x2F900, 0x2F901, 0x2F902, 
	0x2F903, 0x2F904, 0x2F905, 0x2F906, 0x2F907, 0x2F908, 0x2F909, 0x2F90A, 
	0x2F90B, 0x2F90C, 0x2F90D, 0x2F90E, 0x2F90F, 0x2F910, 0x2F911, 0x2F912, 
	0x2F913, 0x2F914, 0x2F915, 0x2F916, 0x2F917, 0x2F918, 0x2F919, 0x2F91A, 
	0x2F91B, 0x2F91C, 0x2F91D, 0x2F91E, 0x2F91F, 0x2F920, 0x2F921, 0x2F922, 
	0x2F923, 0x2F924, 0x2F925, 0x2F926, 0x2F927, 0x2F928, 0x2F929, 0x2F92A, 
	0x2F92B, 0x2F92C, 0x2F92D, 0x2F92E, 0x2F92F, 0x2F930, 0x2F931, 0x2F932, 
	0x2F933, 0x2F934, 0x2F935, 0x2F936, 0x2F937, 0x2F938, 0x2F939, 0x2F93A, 
	0x2F93B, 0x2F93C, 0x2F93D, 0x2F93E, 0x2F93F, 0x2F940, 0x2F941, 0x2F942, 
	0x2F943, 0x2F944, 0x2F945, 0x2F946, 0x2F947, 0x2F948, 0x2F949, 0x2F94A, 
	0x2F94B, 0x2F94C, 0x2F94D, 0x2F94E, 0x2F94F, 0x2F950, 0x2F951, 0x2F952, 
	0x2F953, 0x2F954, 0x2F955, 0x2F956, 0x2F957, 0x2F958, 0x2F959, 0x2F95A, 
	0x2F95B, 0x2F95C, 0x2F95D, 0x2F95E, 0x2F95F, 0x2F960, 0x2F961, 0x2F962, 
	0x2F963, 0x2F964, 0x2F965, 0x2F966, 0x2F967, 0x2F968, 0x2F969, 0x2F96A, 
	0x2F96B, 0x2F96C, 0x2F96D, 0x2F96E, 0x2F96F, 0x2F970, 0x2F971, 0x2F972, 
	0x2F973, 0x2F974, 0x2F975, 0x2F976, 0x2F977, 0x2F978, 0x2F979, 0x2F97A, 
	0x2F97B, 0x2F97C, 0x2F97D, 0x2F97E, 0x2F97F, 0x2F980, 0x2F981, 0x2F982, 
	0x2F983, 0x2F984, 0x2F985, 0x2F986, 0x2F987, 0x2F988, 0x2F989, 0x2F98A, 
	0x2F98B, 0x2F98C, 0x2F98D, 0x2F98E, 0x2F98F, 0x2F990, 0x2F991, 0x2F992, 
	0x2F993, 0x2F994, 0x2F995, 0x2F996, 0x2F997, 0x2F998, 0x2F999, 0x2F99A, 
	0x2F99B, 0x2F99C, 0x2F99D, 0x2F99E, 0x2F99F, 0x2F9A0, 0x2F9A1, 0x2F9A2, 
	0x2F9A3, 0x2F9A4, 0x2F9A5, 0x2F9A6, 0x2F9A7, 0x2F9A8, 0x2F9A9, 0x2F9AA, 
	0x2F9AB, 0x2F9AC, 0x2F9AD, 0x2F9AE, 0x2F9AF, 0x2F9B0, 0x2F9B1, 0x2F9B2, 
	0x2F9B3, 0x2F9B4, 0x2F9B5, 0x2F9B6, 0x2F9B7, 0x2F9B8, 0x2F9B9, 0x2F9BA, 
	0x2F9BB, 0x2F9BC, 0x2F9BD, 0x2F9BE, 0x2F9BF, 0x2F9C0, 0x2F9C1, 0x2F9C2, 
	0x2F9C3, 0x2F9C4, 0x2F9C5, 0x2F9C6, 0x2F9C7, 0x2F9C8, 0x2F9C9, 0x2F9CA, 
	0x2F9CB, 0x2F9CC, 0x2F9CD, 0x2F9CE, 0x2F9CF, 0x2F9D0, 0x2F9D1, 0x2F9D2, 
	0x2F9D3, 0x2F9D4, 0x2F9D5, 0x2F9D6, 0x2F9D7, 0x2F9D8, 0x2F9D9, 0x2F9DA, 
	0x2F9DB, 0x2F9DC, 0x2F9DD, 0x2F9DE, 0x2F9DF, 0x2F9E0, 0x2F9E1, 0x2F9E2, 
	0x2F9E3, 0x2F9E4, 0x2F9E5, 0x2F9E6, 0x2F9E7, 0x2F9E8, 0x2F9E9, 0x2F9EA, 
	0x2F9EB, 0x2F9EC, 0x2F9ED, 0x2F9EE, 0x2F9EF, 0x2F9F0, 0x2F9F1, 0x2F9F2, 
	0x2F9F3, 0x2F9F4, 0x2F9F5, 0x2F9F6, 0x2F9F7, 0x2F9F8, 0x2F9F9, 0x2F9FA, 
	0x2F9FB, 0x2F9FC, 0x2F9FD, 0x2F9FE, 0x2F9FF, 0x2FA00, 0x2FA01, 0x2FA02, 
	0x2FA03, 0x2FA04, 0x2FA05, 0x2FA06, 0x2FA07, 0x2FA08, 0x2FA09, 0x2FA0A, 
	0x2FA0B, 0x2FA0C, 0x2FA0D, 0x2FA0E, 0x2FA0F, 0x2FA10, 0x2FA11, 0x2FA12, 
	0x2FA13, 0x2FA14, 0x2FA15, 0x2FA16, 0x2FA17, 0x2FA18, 0x2FA19, 0x2FA1A, 
	0x2FA1B, 0x2FA1C, 0x2FA1D, 0xE0001, 0xE0020, 0xE0021, 0xE0022, 0xE0023, 
	0xE0024, 0xE0025, 0xE0026, 0xE0027, 0xE0028, 0xE0029, 0xE002A, 0xE002B, 
	0xE002C, 0xE002D, 0xE002E, 0xE002F, 0xE0030, 0xE0031, 0xE0032, 0xE0033, 
	0xE0034, 0xE0035, 0xE0036, 0xE0037, 0xE0038, 0xE0039, 0xE003A, 0xE003B, 
	0xE003C, 0xE003D, 0xE003E, 0xE003F, 0xE0040, 0xE0041, 0xE0042, 0xE0043, 
	0xE0044, 0xE0045, 0xE0046, 0xE0047, 0xE0048, 0xE0049, 0xE004A, 0xE004B, 
	0xE004C, 0xE004D, 0xE004E, 0xE004F, 0xE0050, 0xE0051, 0xE0052, 0xE0053, 
	0xE0054, 0xE0055, 0xE0056, 0xE0057, 0xE0058, 0xE0059, 0xE005A, 0xE005B, 
	0xE005C, 0xE005D, 0xE005E, 0xE005F, 0xE0060, 0xE0061, 0xE0062, 0xE0063, 
	0xE0064, 0xE0065, 0xE0066, 0xE0067, 0xE0068, 0xE0069, 0xE006A, 0xE006B, 
	0xE006C, 0xE006D, 0xE006E, 0xE006F, 0xE0070, 0xE0071, 0xE0072, 0xE0073, 
	0xE0074, 0xE0075, 0xE0076, 0xE0077, 0xE0078, 0xE0079, 0xE007A, 0xE007B, 
	0xE007C, 0xE007D, 0xE007E, 0xE007F, 0xE0100, 0xE0101, 0xE0102, 0xE0103, 
	0xE0104, 0xE0105, 0xE0106, 0xE0107, 0xE0108, 0xE0109, 0xE010A, 0xE010B, 
	0xE010C, 0xE010D, 0xE010E, 0xE010F, 0xE0110, 0xE0111, 0xE0112, 0xE0113, 
	0xE0114, 0xE0115, 0xE0116, 0xE0117, 0xE0118, 0xE0119, 0xE011A, 0xE011B, 
	0xE011C, 0xE011D, 0xE011E, 0xE011F, 0xE0120, 0xE0121, 0xE0122, 0xE0123, 
	0xE0124, 0xE0125, 0xE0126, 0xE0127, 0xE0128, 0xE0129, 0xE012A, 0xE012B, 
	0xE012C, 0xE012D, 0xE012E, 0xE012F, 0xE0130, 0xE0131, 0xE0132, 0xE0133, 
	0xE0134, 0xE0135, 0xE0136, 0xE0137, 0xE0138, 0xE0139, 0xE013A, 0xE013B, 
	0xE013C, 0xE013D, 0xE013E, 0xE013F, 0xE0140, 0xE0141, 0xE0142, 0xE0143, 
	0xE0144, 0xE0145, 0xE0146, 0xE0147, 0xE0148, 0xE0149, 0xE014A, 0xE014B, 
	0xE014C, 0xE014D, 0xE014E, 0xE014F, 0xE0150, 0xE0151, 0xE0152, 0xE0153, 
	0xE0154, 0xE0155, 0xE0156, 0xE0157, 0xE0158, 0xE0159, 0xE015A, 0xE015B, 
	0xE015C, 0xE015D, 0xE015E, 0xE015F, 0xE0160, 0xE0161, 0xE0162, 0xE0163, 
	0xE0164, 0xE0165, 0xE0166, 0xE0167, 0xE0168, 0xE0169, 0xE016A, 0xE016B, 
	0xE016C, 0xE016D, 0xE016E, 0xE016F, 0xE0170, 0xE0171, 0xE0172, 0xE0173, 
	0xE0174, 0xE0175, 0xE0176, 0xE0177, 0xE0178, 0xE0179, 0xE017A, 0xE017B, 
	0xE017C, 0xE017D, 0xE017E, 0xE017F, 0xE0180, 0xE0181, 0xE0182, 0xE0183, 
	0xE0184, 0xE0185, 0xE0186, 0xE0187, 0xE0188, 0xE0189, 0xE018A, 0xE018B, 
	0xE018C, 0xE018D, 0xE018E, 0xE018F, 0xE0190, 0xE0191, 0xE0192, 0xE0193, 
	0xE0194, 0xE0195, 0xE0196, 0xE0197, 0xE0198, 0xE0199, 0xE019A, 0xE019B, 
	0xE019C, 0xE019D, 0xE019E, 0xE019F, 0xE01A0, 0xE01A1, 0xE01A2, 0xE01A3, 
	0xE01A4, 0xE01A5, 0xE01A6, 0xE01A7, 0xE01A8, 0xE01A9, 0xE01AA, 0xE01AB, 
	0xE01AC, 0xE01AD, 0xE01AE, 0xE01AF, 0xE01B0, 0xE01B1, 0xE01B2, 0xE01B3, 
	0xE01B4, 0xE01B5, 0xE01B6, 0xE01B7, 0xE01B8, 0xE01B9, 0xE01BA, 0xE01BB, 
	0xE01BC, 0xE01BD, 0xE01BE, 0xE01BF, 0xE01C0, 0xE01C1, 0xE01C2, 0xE01C3, 
	0xE01C4, 0xE01C5, 0xE01C6, 0xE01C7, 0xE01C8, 0xE01C9, 0xE01CA, 0xE01CB, 
	0xE01CC, 0xE01CD, 0xE01CE, 0xE01CF, 0xE01D0, 0xE01D1, 0xE01D2, 0xE01D3, 
	0xE01D4, 0xE01D5, 0xE01D6, 0xE01D7, 0xE01D8, 0xE01D9, 0xE01DA, 0xE01DB, 
	0xE01DC, 0xE01DD, 0xE01DE, 0xE01DF, 0xE01E0, 0xE01E1, 0xE01E2, 0xE01E3, 
	0xE01E4, 0xE01E5, 0xE01E6, 0xE01E7, 0xE01E8, 0xE01E9, 0xE01EA, 0xE01EB, 
	0xE01EC, 0xE01ED, 0xE01EE, 0xE01EF, 	
};



//----------------------------------------------------------------------------------------------

/**
   @SYMTestCaseID           TI18N-UCD-CT-4001
   @SYMPREQ                 2471
   @SYMTestCaseDesc         Check UCD support for newly added codepoint.
   @SYMTestPriority         High 
   @SYMTestStatus           Implemented
   @SYMTestActions          1. Select new UCD codepoint
                            2. Get the characters' properties from UCD
   @SYMTestExpectedResults  All properties of the characters are retrieved from UCD
 */
//----------------------------------------------------------------------------------------------
// test for characters' attribute newly added or changed in Unicode 5.0
// the characters are chosen randomly from different blocks
void test_unicode5_new_chars()
	{
	test_unicode_char(0x002B,_L("Sm b=ES w=3"),_L("+"));
	test_unicode_char(0x002D,_L("Pd b=ES w=3"),_L("-"));
	test_unicode_char(0x002F,_L("Po b=CS w=3"),_L("/"));
	test_unicode_char(0x00AD,_L("Cf b=BN np ng"),_L("SOFT HYPHEN"));
	test_unicode_char(0x00B2,_L("No b=EN n=2"),_L("SUPERSCRIPT TWO"));
	test_unicode_char(0x019E,_L("Ll u=220"),_L("LATIN SMALL LETTER N WITH LONG RIGHT LEG"));
	test_unicode_char(0x01C5,_L("Lt u=1c4 l=1c6"),_L("LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON"));
	test_unicode_char(0x0234,_L("Ll"),_L("LATIN SMALL LETTER L WITH CURL"));
	test_unicode_char(0x0294,_L("Lo"),_L("LATIN LETTER GLOTTAL STOP"));
	test_unicode_char(0x02AE,_L("Ll"),_L("LATIN SMALL LETTER TURNED H WITH FISHHOOK"));
	test_unicode_char(0x02C6,_L("Lm b=ON"),_L("MODIFIER LETTER CIRCUMFLEX ACCENT"));
	test_unicode_char(0x02FF,_L("Sk b=ON"),_L("MODIFIER LETTER LOW LEFT ARROW"));
	test_unicode_char(0x034F,_L("Mn b=NSM"),_L("COMBINING GRAPHEME JOINER"));
	test_unicode_char(0x03D8,_L("Lu l=3d9"),_L("GREEK LETTER ARCHAIC KOPPA"));
	test_unicode_char(0x03F2,_L("Ll u=3f9"),_L("GREEK LUNATE SIGMA SYMBOL"));
	test_unicode_char(0x04C0,_L("Lu l=4cf"),_L("CYRILLIC LETTER PALOCHKA"));
	test_unicode_char(0x06FF,_L("Lo b=AL"),_L("ARABIC LETTER HEH WITH INVERTED V"));
	test_unicode_char(0x07C0,_L("Nd b=R n=0"),_L("NKO DIGIT ZERO"));
	test_unicode_char(0x097F,_L("Lo"),_L("DEVANAGARI LETTER BBA"));
	test_unicode_char(0x0B83,_L("Lo"),_L("TAMIL SIGN VISARGA"));
	test_unicode_char(0x0CBD,_L("Lo"),_L("KANNADA SIGN AVAGRAHA"));
	test_unicode_char(0x0F3A,_L("Ps b=ON m"),_L("TIBETAN MARK GUG RTAGS GYON"));
	test_unicode_char(0x10A0,_L("Lu l=2d00"),_L("GEORGIAN CAPITAL LETTER AN"));
	test_unicode_char(0x1369,_L("No n=1"),_L("ETHIOPIC DIGIT ONE"));
	test_unicode_char(0x1900,_L("Lo"),_L("LIMBU VOWEL-CARRIER LETTER"));
	test_unicode_char(0x200B,_L("Cf b=BN np ng"),_L("ZERO WIDTH SPACE"));
	test_unicode_char(0x202F,_L("Zs b=CS ng"),_L("NARROW NO-BREAK SPACE"));
	test_unicode_char(0x214E,_L("Ll u=2132"),_L("TURNED SMALL F"));
	test_unicode_char(0x2DBE,_L("Lo"),_L("ETHIOPIC SYLLABLE CCHO"));
	test_unicode_char(0x4DC0,_L("So b=ON w=4"),_L("HEXAGRAM FOR THE CREATIVE HEAVEN"));
	test_unicode_char(0x9FA6,_L("Lo w=4"),_L("CJK UNIFIED IDEOGRAPH-9FA6"));
	test_unicode_char(0xA721,_L("Sk b=ON w=4"),_L("MODIFIER LETTER STRESS AND LOW TONE"));
	test_unicode_char(0xFE00,_L("Mn b=NSM"),_L("VARIATION SELECTOR-1"));
	test_unicode_char(0x10100,_L("Po"),_L("AEGEAN WORD SEPARATOR LINE"));
	test_unicode_char(0x20000,_L("Lo w=4"),_L("CJK UNIFIED IDEOGRAPH-20000"));
	test_unicode_char(0x2F800,_L("Lo w=4 d=4e3d"),_L("CJK UNIFIED IDEOGRAPH-2F800"));
	}

static void Unicode5NewCharTest()
	{
	TUint32 i;
	for (i=0; i < sizeof(Unicode5NewCharRange)/sizeof(TUint32); i+=2)
		{
		TUint32 from = Unicode5NewCharRange[i];
		TUint32 to = Unicode5NewCharRange[i + 1];
		
		for (TUint32 ch = from; ch <= to; ch++)
			{
			TChar c(ch);

			TChar::TCharInfo ci;
			c.GetInfo(ci);
		
			test((ci.iCategory & TChar::EUnassignedGroup) != TChar::EUnassignedGroup);
			
			}
		}
	for (i=0; i < sizeof(Unicode5NewCharOutBMPIndex)/sizeof(TUint32); i++)
		{
		TUint32 ch = Unicode5NewCharOutBMPIndex[i]; 
		TChar c(ch);

		TChar::TCharInfo ci;
		c.GetInfo(ci);
	
		test((ci.iCategory & TChar::EUnassignedGroup) != TChar::EUnassignedGroup);

		}
	}

//----------------------------------------------------------------------------------------------

/**
   @SYMTestCaseID           TI18N-UCD-CT-4002
   @SYMPREQ                 2471
   @SYMTestCaseDesc         Check UCD support for one byte GB18030 character mapping/support.
   @SYMTestPriority         High 
   @SYMTestStatus           Implemented
   @SYMTestActions          1. Select characters from one-byte character set
                            2. Get the characters' properties from UCD
   @SYMTestExpectedResults  All properties of the characters are retrieved from UCD
 */
//----------------------------------------------------------------------------------------------
static void GB18030OneByteCharTest()
	{
	for (TInt32 i=0; i <= 127; i++)
		{
		TUint32 ch = i;
		TChar c(ch);

		TChar::TCharInfo ci;
		c.GetInfo(ci);
	
		test((ci.iCategory & TChar::EUnassignedGroup) != TChar::EUnassignedGroup);
		}
	}

//----------------------------------------------------------------------------------------------

/**
   @SYMTestCaseID           TI18N-UCD-CT-4003
   @SYMPREQ                 2471
   @SYMTestCaseDesc         Check UCD support for four-byte GB18030 character mapping/support.
   @SYMTestPriority         High 
   @SYMTestStatus           Implemented
   @SYMTestActions          1. Select characters from CJK Ext B
                            2. Get the characters' properties from UCD
                            3. Select characters from CJK Compat Ideograph
                            4. Get the character's properties from UCD
   @SYMTestExpectedResults  All properties of the characters are retrieved from UCD
 */
//----------------------------------------------------------------------------------------------
static void GB18030FourBytesCharTest()
	{
	/* CJK Ideograph Extension B */
	for (TInt32 i=0x20000; i <= 0x2A6D6; i++)
		{
		TUint32 ch = i;
		TChar c(ch);

		TChar::TCharInfo ci;
		c.GetInfo(ci);
	
		test((ci.iCategory & TChar::EUnassignedGroup) != TChar::EUnassignedGroup);
		}
	
	/* CJK COMPATIBILITY IDEOGRAPH has been tested in Unicode 5 New char */
	}

GLDEF_C TInt E32Main(void)
    {

	test.Title();
	test.Start(_L("TCHAR TESTING"));

	test_unicode_chars();
	test_unicode_collations();

	test.Next(_L("TChar"));
	TestTChar<TChar> t;
	t.Test1();
	t.Test2(eTChar);
	t.Test3(eTChar);
	t.Test4(eTChar);

	test.Next(_L("TCharF"));
	TestTChar<TCharF> f;
	f.Test2(eTCharF);
	f.Test3(eTCharF);
	f.Test4(eTCharF);

	test.Next(_L("TCharLC"));
	TestTChar<TCharLC> lc;
	lc.Test2(eTCharLC);
	lc.Test3(eTCharLC);
	lc.Test4(eTCharLC);

	test.Next(_L("TCharUC"));
  	TestTChar<TCharUC> uc;
	uc.Test2(eTCharUC);
	uc.Test3(eTCharUC);
	uc.Test4(eTCharUC);

	test.Next(_L("Unicode 5 New"));
	Unicode5NewCharTest();
	test_unicode5_new_chars();
	
	test.Next(_L("GB18030 One and Four Bytes Character"));
	GB18030OneByteCharTest();
	GB18030FourBytesCharTest();
	
	test.End();
	return(KErrNone);
    }

#pragma warning ( default :4700)
#pragma warning ( default :4710) 
