@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15

# Copyright (C) 2005, 2006 Apple Computer, Inc.  All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1.  Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer. 
# 2.  Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution. 
# 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
#     its contributors may be used to endorse or promote products derived
#     from this software without specific prior written permission. 
#
# THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# "patch" script for WebKit Open Source Project, used to apply patches.

# Differences from invoking "patch -p0":
#
#   Handles added files (does a svn add).
#   Handles added directories (does a svn add).
#   Handles removed files (does a svn rm).
#   Handles removed directories--those with no more files or directories left in them
#       (does a svn rm).
#   Has mode where it will roll back to svn version numbers in the patch file so svn
#       can do a 3-way merge.
#   Paths from Index: lines are used rather than the paths on the patch lines, which
#       makes patches generated by "cvs diff" work (increasingly unimportant since we
#       use Subversion now).
#   ChangeLog patches use --fuzz=3 to prevent rejects.
#   Handles binary files (requires patches made by svn-create-patch).
#
# Missing features:
#
#   Handle property changes.
#   Handle file moves (would require patches made by svn-create-patch).
#   When doing a removal, check that old file matches what's being removed.
#   Notice a patch that's being applied at the "wrong level" and make it work anyway.
#   Do a dry run on the whole patch and don't do anything if part of the patch is
#       going to fail (probably too strict unless we exclude ChangeLog).

use strict;
use warnings;

use Cwd;
use File::Basename;
use File::Spec;
use Getopt::Long;
use MIME::Base64;
use FindBin qw($Bin);

sub addDirectoriesIfNeeded($);
sub applyPatch($$;$);
sub handleBinaryChange($$);
sub isDirectoryEmptyForRemoval($);
sub patch($);
sub removeDirectoriesIfNeeded();
sub svnStatus($);

my $patch_cmd = "$Bin\\patch.exe";
my $merge = 0;
my $showHelp = 0;
if (!GetOptions("merge!" => \$merge, "help!" => \$showHelp) || $showHelp) {
    print STDERR basename($0) . " [-h|--help] [-m|--merge] patch1 [patch2 ...]\n";
    exit 1;
}

my %removeDirectoryIgnoreList = (
    '.' => 1,
    '..' => 1,
    '.svn' => 1,
    '_svn' => 1,
);

my %checkedDirectories;
my @patches;
my %versions;

my $indexPath;
my $patch;
while (<>) {
    s/\r//g;
    chomp;
    if (/^Index: (.+)/) {
        $indexPath = $1;
        if ($patch) {
            push @patches, $patch;
            $patch = "";
        }
    }
    if ($indexPath) {
        # Fix paths on diff, ---, and +++ lines to match preceding Index: line.
        s/\S+$/$indexPath/ if /^diff/;
        s/^--- \S+/--- $indexPath/;
        if (s/^\+\+\+ \S+/+++ $indexPath/) {
            $indexPath = "";
        }
    }
    if (/^--- .+\(revision (\d+)\)$/) {
        $versions{$indexPath} = $1 if( $1 != 0 );
    }
    $patch .= $_;
    $patch .= "\n";
}

push @patches, $patch if $patch;

if ($merge) {
    for my $file (sort keys %versions) {
        print "Getting version $versions{$file} of $file\n";
        system "svn", "update", "-r", $versions{$file}, $file;
    }
}

for $patch (@patches) {
    patch($patch);
}

removeDirectoriesIfNeeded();

exit 0;

sub addDirectoriesIfNeeded($)
{
    my ($path) = @_;
    my @dirs = File::Spec->splitdir($path);
    my $dir = ".";
    while (scalar @dirs) {
        $dir = File::Spec->catdir($dir, shift @dirs);
        next if exists $checkedDirectories{$dir};
        if (! -e $dir) {
            mkdir $dir or die "Failed to create required directory '$dir' for path '$path'\n";
            system "svn", "add", $dir;
            $checkedDirectories{$dir} = 1;
        }
        elsif (-d $dir) {
            my $svnOutput = svnStatus($dir);
            if ($svnOutput && substr($svnOutput, 0, 1) eq "?") {
                system "svn", "add", $dir;
            }
            $checkedDirectories{$dir} = 1;
        }
        else {
            die "'$dir' is not a directory";
        }
    }
}

sub applyPatch($$;$)
{
    my ($patch, $fullPath, $options) = @_;
    $options = [] if (! $options);
    my $command = "$patch_cmd " . join(" ", "-p0", @{$options});
    open PATCH, "| $command" or die "Failed to patch $fullPath\n";
    print PATCH $patch;
    close PATCH;
}

sub handleBinaryChange($$)
{
    my ($fullPath, $contents) = @_;
    if ($contents =~ m#((\n[A-Za-z0-9+/]{76})+\n[A-Za-z0-9+/=]{4,76}\n)\n#) {
        # Addition or Modification
        open FILE, ">", $fullPath or die;
        print FILE decode_base64($1);
        close FILE;
        my $svnOutput = svnStatus($fullPath);
        if ($svnOutput && substr($svnOutput, 0, 1) eq "?") {
            # Addition
            system "svn", "add", $fullPath;
        } else {
            # Modification
            print $svnOutput if $svnOutput;
        }
    } else {
        # Deletion
        system "svn", "rm", $fullPath;
    }
}

sub isDirectoryEmptyForRemoval($)
{
    my ($dir) = @_;
    my $directoryIsEmpty = 1;
    opendir DIR, $dir or die "Could not open '$dir' to list files: $?";
    for (my $item = readdir DIR; $item && $directoryIsEmpty; $item = readdir DIR) {
        next if exists $removeDirectoryIgnoreList{$item};
        if (! -d File::Spec->catdir($dir, $item)) {
            $directoryIsEmpty = 0;
        } else {
            my $svnOutput = svnStatus(File::Spec->catdir($dir, $item));
            next if $svnOutput && substr($svnOutput, 0, 1) eq "D";
            $directoryIsEmpty = 0;
        }
    }
    closedir DIR;
    return $directoryIsEmpty;
}

sub patch($)
{
    my ($patch) = @_;
    return if !$patch;

    $patch =~ m|^Index: ([^\n]+)| or die "Failed to find 'Index:' in \"$patch\"\n";
    my $fullPath = $1;

    my $deletion = 0;
    my $addition = 0;
    my $isBinary = 0;

    $addition = 1 if $patch =~ /\n--- .+\(revision 0\)\n/;
    $deletion = 1 if $patch =~ /\n@@ .* \+0,0 @@/;
    $isBinary = 1 if $patch =~ /\nCannot display: file marked as a binary type\./;

    if (!$addition && !$deletion && !$isBinary) {
        # Standard patch, patch tool can handle this.
        if (basename($fullPath) eq "ChangeLog") {
            my $changeLogDotOrigExisted = -f "${fullPath}.orig";
            applyPatch($patch, $fullPath, ["--fuzz=3"]);
            unlink("${fullPath}.orig") if (! $changeLogDotOrigExisted);
        } else {
            applyPatch($patch, $fullPath);
        }
    } else {
        # Either a deletion, an addition or a binary change.

        addDirectoriesIfNeeded(dirname($fullPath));

        if ($isBinary) {
            # Binary change
            handleBinaryChange($fullPath, $patch);
        } elsif ($deletion) {
            # Deletion.
            system "svn", "rm", $fullPath;
        } else {
            # Addition.
            my $contents = $patch;
            if ($contents !~ s/^(.*\n)*@@[^\n]+@@\n//) {
                # Empty contents.
                $contents = "";
            } else {
                # Non-empty contents: Remove leading + signs.
                $contents =~ s/^\+//;
                $contents =~ s/\n\+/\n/g;
            }
            open FILE, ">", $fullPath or die;
            print FILE $contents;
            close FILE;
            system "svn", "add", $fullPath;
        }
    }
}

sub removeDirectoriesIfNeeded()
{
    foreach my $dir (reverse sort keys %checkedDirectories) {
        if (isDirectoryEmptyForRemoval($dir)) {
            my $svnOutput;
            open SVN, "svn rm $dir |" or die;
            # Only save the last line since Subversion lists all changed statuses below $dir
            while (<SVN>) {
                $svnOutput = $_;
            }
            close SVN;
            print $svnOutput if $svnOutput;
        }
    }
}

sub svnStatus($)
{
    my ($fullPath) = @_;
    open SVN, "svn status --non-interactive --non-recursive $fullPath |" or die;
    my $svnStatus = <SVN>;
    close SVN;
    return $svnStatus;
}

__END__
:endofperl
