/*
 * Copyright (C) 2007 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer. 
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
#ifndef IconRecord_h
#define IconRecord_h

#include "PageURLRecord.h"
#include "Shared.h"
#include "SharedBuffer.h"

#include <wtf/HashSet.h>
#include <wtf/OwnPtr.h>
#include "PlatformString.h"
#include "StringHash.h"

namespace WebCore { 

class IconDataSnapshot;
class Image;
class IntSize;
class SQLDatabase;

enum ImageDataStatus {
    ImageDataStatusPresent, ImageDataStatusMissing, ImageDataStatusUnknown
};

class IconSnapshot {
public:
    IconSnapshot() : timestamp(0) { }
    
    IconSnapshot(const String& url, int stamp, SharedBuffer* theData)
        : iconURL(url)
        , timestamp(stamp)
        , data(theData)
    { }
        
    String iconURL;
    int timestamp;
    RefPtr<SharedBuffer> data;
};

/* To avoid deletion of icon bitmaps in Symbian */    
#if PLATFORM(SYMBIAN)
class IconImagePtr : Noncopyable {
        typedef Image* PtrType;
public:
        explicit IconImagePtr(PtrType ptr = 0) : m_ptr(ptr) { }
        ~IconImagePtr() { /*deleteOwnedPtr(m_ptr);*/ }

        PtrType get() const { return m_ptr; }
        PtrType release() { PtrType ptr = m_ptr; m_ptr = 0; return ptr; }

        void set(PtrType ptr) { ASSERT(!ptr || m_ptr != ptr); /*deleteOwnedPtr(m_ptr);*/ m_ptr = ptr; }
        void clear() { /*deleteOwnedPtr(m_ptr);*/ m_ptr = 0; }

        Image& operator*() const { ASSERT(m_ptr); return *m_ptr; }
        PtrType operator->() const { ASSERT(m_ptr); return m_ptr; }

        bool operator!() const { return !m_ptr; }

        // This conversion operator allows implicit conversion to bool but not to other integer types.
        typedef PtrType (IconImagePtr::*UnspecifiedBoolType)() const;
        operator UnspecifiedBoolType() const { return m_ptr ? &IconImagePtr::get : 0; }

        void swap(IconImagePtr& o) { std::swap(m_ptr, o.m_ptr); }

    private:
        PtrType m_ptr;
         };

    inline void swap(IconImagePtr& a, IconImagePtr& b) { a.swap(b); }
    inline Image* getPtr(const IconImagePtr& p)
    {
        return p.get();
    }
#endif

class IconRecord : public Shared<IconRecord> {
    friend class PageURLRecord;
public:
    IconRecord(const String& url); 
    ~IconRecord();
    
    time_t getTimestamp() { return m_stamp; }
    void setTimestamp(time_t stamp) { m_stamp = stamp; }
        
    void setImageData(PassRefPtr<SharedBuffer> data);
    Image* image(const IntSize&);    
    
    String iconURL() { return m_iconURL; }

    void loadImageFromResource(const char*);
        
    ImageDataStatus imageDataStatus();
    
    const HashSet<String>& retainingPageURLs() { return m_retainingPageURLs; }
    
    IconSnapshot snapshot(bool forDeletion = false) const;
private:
    String m_iconURL;
    time_t m_stamp;
#if PLATFORM(SYMBIAN)
    IconImagePtr m_image;
#else    
    OwnPtr<Image> m_image;
#endif
    
    HashSet<String> m_retainingPageURLs;
        
    // This allows us to cache whether or not a SiteIcon has had its data set yet
    // This helps the IconDatabase know if it has to set the data on a new object or not,
    // and also to determine if the icon is missing data or if it just hasn't been brought
    // in from the DB yet
    bool m_dataSet;
    
    // FIXME - Right now WebCore::Image doesn't have a very good API for accessing multiple representations
    // Even the NSImage way of doing things that we do in WebKit isn't very clean...  once we come up with a 
    // better way of handling that, we'll likely have a map of size-to-images similar to below
    // typedef HashMap<IntSize, Image*> SizeImageMap;
    // SizeImageMap m_images;
};


} //namespace WebCore

#endif
