/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/05/18 19:36:21 $
 * $Revision: 1.55.2.4 $
 */

#include <ansi_parms.h>

#ifndef _INSIDE_X87D_C_

#undef	_MSL_USE_INLINE
#define	_MSL_USE_INLINE 1
#undef	_MSL_INLINE
#define	_MSL_INLINE _MSL_DO_NOT_INLINE

#endif	/*_INSIDE_X87D_C_*/

#include <math.h>
#include <math_api.h>
#include <float.h>
#include <fenv.h>
#include <errno.h>
#include <string.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>

#ifndef __MWERKS__
	#define _MSL_COMPILE_FOR_SPEED 1
#else
	#if __option(optimize_for_size)
		#define _MSL_COMPILE_FOR_SPEED 0
	#else
		#define _MSL_COMPILE_FOR_SPEED 1
	#endif
#endif /* __MWERKS__ */

#if _MSL_C99

#if _MSL_FLOATING_POINT

_MSL_IMP_EXP_C float _MSL_MATH_CDECL cbrtf(float x) _MSL_CANT_THROW
{
	double xd;
	unsigned int* ix;
	unsigned int sign;
	int exponent;
	if (x == 0)
		return x;
	ix = (unsigned int*)&x;
/* 	if x is +/- inf or nan, return x, setting errors as appropriate									*/
	if (!isfinite(x))
	{
		if (math_errhandling)
		{
			if (isnan(x))
			{
				if (math_errhandling & MATH_ERRNO)
					errno = EDOM;
				if (math_errhandling & MATH_ERREXCEPT)
					feraiseexcept(FE_INVALID);
			}
			else
			{
				if (math_errhandling & MATH_ERRNO)
					errno = ERANGE;
				if (math_errhandling & MATH_ERREXCEPT)
					feraiseexcept(FE_OVERFLOW);
			}
		}
		return x;
	}

	sign = *ix & 0x80000000;
	*ix ^= sign;
	exponent = (int)(*ix >> 23);
	if (exponent == 0)
	{
		int c = __msl_count_leading_zero32(*ix) - 8;
		*ix <<= c;
		exponent -= c - 1;
	}
	exponent -= 126;
	*ix &= 0x007FFFFF;
	*ix |= 0x3F000000;
	switch (exponent % 3)
	{
		case -2:
		case 1:
			exponent += 2;
			*ix -= 0x01000000;
			break;
		case -1:
		case 2:
			exponent += 1;
			*ix -= 0x00800000;
			break;
	}
	xd = x;
	x = (float)((0.164705865585441 + (13.2513868634597 + (115.1358553761178 +
	                         (181.9414139157457 + 41.95971194004274 * xd) * xd) * xd) * xd) /
	    (1.0               + (33.30169492280659 + (161.6940396106312 +
	                         (142.8167287366127 + 13.64061797885738 * xd) * xd) * xd) * xd));
	exponent /= 3;
	exponent <<= 23;
	*ix += exponent;
	*ix |= sign;
	return x;
}

#endif /* _MSL_FLOATING_POINT */

#if !_MSL_USES_SUN_MATH_LIB

int ilogb(double x)
{
	if (x == 0.0)
		return (FP_ILOGB0);
	if (isinf(x))
		return (INT_MAX);
	if (isnan(x))
		return (FP_ILOGBNAN);
	return ((int)logb(x));
}

int ilogbl(long double x)
{
	if (x == 0.0)
		return (FP_ILOGB0);
	if (isinf(x))
		return (INT_MAX);
	if (isnan(x))
		return (FP_ILOGBNAN);
	return ((int)logbl(x));
}

_MSL_IMP_EXP_C double _MSL_MATH_CDECL cbrt(double x) _MSL_CANT_THROW
{
	unsigned int sign;
	int exponent;
	double y;
/* 		if x is 0 or -0 return x 																	*/
	if (x == 0)
		return x;
/* 	if x is +/- inf or nan, return x, setting errors as appropriate									*/
	if (!isfinite(x))
	{
		if (math_errhandling)
		{
			if (isnan(x))
			{
				if (math_errhandling & MATH_ERRNO)
					errno = EDOM;
				if (math_errhandling & MATH_ERREXCEPT)
					feraiseexcept(FE_INVALID);
			}
			else
			{
				if (math_errhandling & MATH_ERRNO)
					errno = ERANGE;
				if (math_errhandling & MATH_ERREXCEPT)
					feraiseexcept(FE_OVERFLOW);
			}
		}
		return x;
	}

/* 	extract biased exponent (x is either normal or denormal											*/
	sign = __UHI(x) & 0x80000000;
	__UHI(x) ^= sign;
	exponent = (int)(__UHI(x) >> 20);
	if (exponent == 0)  /* if x is denormal															*/
	{
		/* normalize x by shifting left until the first 1 bit is in									*/
		/* position 11																				*/
		int c = __msl_count_leading_zero32(__UHI(x)) - 11;
		if (c == 21)
			c += __msl_count_leading_zero32(__ULO(x));
		if (c < 32)
		{
			__UHI(x) = (__UHI(x) << c) | (__ULO(x) >> (32 - c));
			__ULO(x) <<= c;
		}
		else
		{
			__UHI(x) = __ULO(x) << (c - 32);
			__ULO(x) = 0;
		}
		/* decrement exponent by the number of leading 0 bits in mantissa							*/
		exponent -= c - 1;
	}

	/* unbias exponent and normalize x to [0.5, 1)													*/
	exponent -= 1022;
	__UHI(x) &= 0x000FFFFF;
	__UHI(x) |= 0x3FE00000;
	/* make exponent evenly divisible by 3, normalizing x to [0.125, 1)								*/
	switch (exponent % 3)
	{
		case -2:
		case 1:
			exponent += 2;
			__UHI(x) -= 0x00200000;
			break;
		case -1:
		case 2:
			exponent += 1;
			__UHI(x) -= 0x00100000;
			break;
	}
	/* estimate first guess to cube root of x [0.125, 1) 											*/
	/* use polynomial derived from minimizing the error  from 0.125 to 1                        	*/
	y = (0.1950252994681516 + (8.93386164028292 +  (34.95109317740758 + 14.90468137136715 * x) * x) * x) /
	    (1.0                + (17.79274627019952 + (34.47634312279896 + 5.715646840256109 * x) * x) * x);
	/* perform newton iterations to improve the estimate											*/
	y = (2 * y + x / (y*y)) / 3;
	/* put result of final estimate back in x														*/
	x = (2 * y + x / (y*y)) / 3;
	/* x is now the cube root of [0.125, 1) -> [0.5, 1)												*/
	/* complete by taking the cube root of the exponent, and loading that							*/
	/* exponent into x																				*/
	exponent /= 3;
	exponent <<= 20;
	__UHI(x) += exponent;
	/* restore original sign of x																	*/
	__UHI(x) |= sign; 
	/*x = x * sign; */
	return x;
}

#endif /*!(_MSL_USES_SUN_MATH_LIB) */

#if __dest_os != __mac_os

/* This function is taken from "Handbook of Mathematical Functions"
   from US Department of Commerce                                  */

#if _MSL_FLOATING_POINT
   
_MSL_IMP_EXP_C long double _MSL_MATH_CDECL erfl(long double x) _MSL_CANT_THROW
{
	long double div;
	long double argx;
	long double coefs[6] =
	{
		0.0705230784,
		0.0422820123,
		0.0092705272,
		0.0001520143,
		0.0002765672,
		0.0000430638
	};
	int index;
	
	argx = x;
	div  = 1;
	for (index = 0; index < 6; index++)
	{
		div += argx * coefs[index];
		argx *= x;
	}
	
	for (index = 0; index < 4; index++)
		div *= div;
		
	return(1.0 - 1.0 / div);
}


_MSL_IMP_EXP_C float _MSL_MATH_CDECL erff(float x) _MSL_CANT_THROW
{
	return((float)erfl(x));
}

_MSL_IMP_EXP_C float _MSL_MATH_CDECL erfcf(float x) _MSL_CANT_THROW
{
	return((float)(1.0 - erfl(x)));
}

_MSL_IMP_EXP_C long double _MSL_MATH_CDECL erfcl(long double x) _MSL_CANT_THROW
{
	return((long double)(1.0 - erfl(x)));
}
	
#define _ln2  0.6931471805599453094172321

#if !(__dest_os ==__win32_os)

_MSL_IMP_EXP_C double _MSL_MATH_CDECL exp2(double x) _MSL_CANT_THROW
{
	return(exp(x * _ln2));
}
#endif

_MSL_IMP_EXP_C float _MSL_MATH_CDECL exp2f(float x) _MSL_CANT_THROW
{
	return(exp(x * _ln2));
}

_MSL_IMP_EXP_C long double _MSL_MATH_CDECL exp2l(long double x) _MSL_CANT_THROW
{
	return(exp(x * _ln2));
}

_MSL_IMP_EXP_C float _MSL_MATH_CDECL lgammaf(float x) _MSL_CANT_THROW
{
	return((float)(logl(tgammal(x))));
}
	
_MSL_IMP_EXP_C long double _MSL_MATH_CDECL lgammal(long double x) _MSL_CANT_THROW
{
	return(logl(tgammal(x)));
}

_MSL_IMP_EXP_C float _MSL_MATH_CDECL nanf(const char* str) _MSL_CANT_THROW
{
	char temparg[32] = "NAN(";
	strcat(temparg, str);
	strcat(temparg, ")");
	
	return(strtof(temparg, (char**)NULL));
}

_MSL_IMP_EXP_C long double _MSL_MATH_CDECL nanl(const char* str) _MSL_CANT_THROW
{
	char temparg[32] = "NAN(";
	strcat(temparg, str);
	strcat(temparg, ")");
	
	return(strtold(temparg, (char**)NULL));
}

_MSL_IMP_EXP_C float _MSL_MATH_CDECL nextafterf(float x, float y) _MSL_CANT_THROW
{
	return nexttowardf(x, (long double)y);
}


_MSL_IMP_EXP_C double _MSL_MATH_CDECL nexttoward(double x, long double y) _MSL_CANT_THROW
{
	int increase;
	int positive;
	unsigned long* lx = (unsigned long*)&x;
	unsigned long* ly = (unsigned long*)&y;
#if _MSL_LITTLE_ENDIAN
	unsigned long x_exp = lx[1] & 0x7FF00000;
	if ((ly[1] & 0x7FF00000) == 0x7FF00000 && (ly[1] & 0x000FFFFF) | ly[0] ||
	    x == y)
		return (double)y;
	if (x_exp == 0x7FF00000 && lx[1] & 0x000FFFFF | lx[0])
		return x;
#else
	unsigned long x_exp = lx[0] & 0x7FF00000;
	if ((ly[0] & 0x7FF00000) == 0x7FF00000 && (ly[0] & 0x000FFFFF) | ly[1] ||
	    x == y)
		return (double)y;
	if (x_exp == 0x7FF00000 && lx[0] & 0x000FFFFF | lx[1])
		return x;
#endif /* _MSL_LITTLE_ENDIAN */
	increase = y > x;
	positive = x > 0;
	if (x_exp != 0x7FF00000 || increase != positive)
	{
		unsigned long long* llx = (unsigned long long*)&x;
		if (x == 0)
		{
			if (increase)
				*llx = 0x0000000000000001ULL;
			else
				*llx = 0x8000000000000001ULL;
		}
		else
		{
			if (increase == positive)
				++(*llx);
			else
				--(*llx);
		}
	#if _MSL_LITTLE_ENDIAN
		x_exp = lx[1] & 0x7FF00000;
	#else
		x_exp = lx[0] & 0x7FF00000;
	#endif /* _MSL_LITTLE_ENDIAN */
		if (x_exp == 0 && (math_errhandling & MATH_ERREXCEPT))
			feraiseexcept(FE_UNDERFLOW | FE_INEXACT);
		else if (x_exp == 0x7FF00000)
		{
			if (math_errhandling & MATH_ERRNO)
				errno = ERANGE;
			if (math_errhandling & MATH_ERREXCEPT)
				feraiseexcept(FE_OVERFLOW | FE_INEXACT);
		}
	}
	return x;
}

_MSL_IMP_EXP_C float _MSL_MATH_CDECL nexttowardf(float x, long double y) _MSL_CANT_THROW
{
	int increase;
	int positive;
	unsigned long* lx = (unsigned long*)&x;
	unsigned long* ly = (unsigned long*)&y;
	unsigned long x_exp = *lx & 0x7F800000;
#if _MSL_LITTLE_ENDIAN
	if ((ly[1] & 0x7FF00000) == 0x7FF00000 && (ly[1] & 0x000FFFFF) | ly[0] ||
	    x == y)
		return (float)y;
#else
	if ((ly[0] & 0x7FF00000) == 0x7FF00000 && (ly[0] & 0x000FFFFF) | ly[1] ||
	    x == y)
		return (float)y;
#endif /* _MSL_LITTLE_ENDIAN */
	if (x_exp == 0x7F800000 && *lx & 0x007FFFFF)
		return x;
	increase = y > x;
	positive = x > 0;
	if (x_exp != 0x7F800000 || increase != positive)
	{
		if (x == 0)
		{
			if (increase)
				*lx = 0x00000001UL;
			else
				*lx = 0x80000001UL;
		}
		else
		{
			if (increase == positive)
				++(*lx);
			else
				--(*lx);
		}
		x_exp = *lx & 0x7F800000;
		if (x_exp == 0 && (math_errhandling & MATH_ERREXCEPT))
			feraiseexcept(FE_UNDERFLOW | FE_INEXACT);
		else if (x_exp == 0x7F800000)
		{
			if (math_errhandling & MATH_ERRNO)
				errno = ERANGE;
			if (math_errhandling & MATH_ERREXCEPT)
				feraiseexcept(FE_OVERFLOW | FE_INEXACT);
		}
	}
	return x;
}

_MSL_IMP_EXP_C double _MSL_MATH_CDECL tgamma(double x) _MSL_CANT_THROW
{
	return((double)tgammal(x));
}


_MSL_IMP_EXP_C float _MSL_MATH_CDECL tgammaf(float x) _MSL_CANT_THROW
{
	return((float)tgammal(x));
}


/* This function is taken from "Handbook of Mathematical Functions"
   from US Department of Commerce                                  */
   
_MSL_IMP_EXP_C long double _MSL_MATH_CDECL tgammal(long double x) _MSL_CANT_THROW
{
	long double coefs[26] =
	{
		+1.0000000000000000,
		+0.5772156649915329,
		-0.6558780715202538,
		-0.0420026350340952,
		+0.1665386113822915,
		-0.0421977345555443,
		-0.0096219715278770,
		+0.0072189432466630,
		-0.0011651675918591,
		-0.0002152416741149,
		+0.0001280502823882,
		-0.0000201348547807,
		-0.0000012504934821,
		+0.0000011330272320,
		-0.0000002056338417,
		+0.0000000061160960,
		+0.0000000050020075,
		-0.0000000011812746,
		+0.0000000001043427,
		+0.0000000000077823,
		-0.0000000000036968,
		+0.0000000000005100,
		-0.0000000000000206,
		-0.0000000000000054,
		+0.0000000000000014,
		+0.0000000000000001
	};
	
	long double argx;
	long double div;
	int    index;
	
	div  = 0.0;
	argx = x;
	for (index = 0; index < 26; index++)
	{
		div += argx * coefs[index];
		argx *= x;
	}
	
	return(1.0/div);
}
#endif /* _MSL_FLOATING_POINT */

#if !_MSL_USES_SUN_MATH_LIB

_MSL_IMP_EXP_C double _MSL_MATH_CDECL erf(double x) _MSL_CANT_THROW
{
	return((double)erfl(x));
}

_MSL_IMP_EXP_C double _MSL_MATH_CDECL erfc(double x) _MSL_CANT_THROW
{
	return((double)(1.0 - erfl(x)));
}

_MSL_IMP_EXP_C double _MSL_MATH_CDECL lgamma(double x) _MSL_CANT_THROW
{
	return((double)(logl(tgammal(x))));
}

_MSL_IMP_EXP_C double _MSL_MATH_CDECL nextafter(double x, double y) _MSL_CANT_THROW
{
	return nexttoward(x, (long double)y);
}

#endif /* _MSL_USES_SUN_MATH_LIB */

#endif /* _MSL_C99 */

#endif /* #if __dest_os == __mac_os */
	

/* Change record: 
 * cc  020124 Needed for __fpclassifyf and other _MSL_INLINE items
 * JWW 020212 Removed the unnecessary __cdecl stuff
 * JWW 020308 Use new _MSL_DO_NOT_INLINE macro for bringing inlined functions into the library
 * mm  021108 Added erf, nan and gamma functions
 * mm  021120 Added nexttowardl()
 * cc  021121 Removed redefined of nexttowardl.  It lives as an inline in math_XXX.h.
 * cc  021216 Added cbrt and family for Michael
 * cc  030110 Fixed so it can be added to mac os and not affect math lib
 * mm  030128 Revised cbrt() and cbrtf() based on Howard's version
 * cc  030221 Added _MSL_USES_SUN_MATH_LIB flag
 * mm  030520 Added C99 wrappers
 * JWW 030321 Added fma, fmaf, and assortied helper functions (thanks to Howard)
 * mm  030522 Added C99 wrappers
 * mm  030529 Added modf() and modff()
 * mm  030605 Added ilogb() and ilogbl()
 * ejs 030613 Standardize the __builtin_xxx types
 * cc  031019 Added Michael's round, roundf,and roundl for sun math
 * cc  030924 Added Michael's lround, lroundf, lroundl, llround, llroundf, 
 *            and llroundl for sun math
 * cc  031003 Added Michael's lrint,lrint and lrintl for sun math
 * cc  031029 Added Michael's nearbyintl, nearbyintf, nearbyint, remquo,
 *            remquol, and remquof for sun math
 * cc  031029 Changed return types of nearbyint, nearbyintf, and nearbyintl
 * cc  040204 Changed float fracpart to long double fracpart in long double versions of 
 *			  nearbyint, lrint, lround, and llround. MTWX10358
 * hh  040410 Added several more intrinsics, changed implementation of __msl_generic_count_leading_zero32
 *            and __msl_generic_count_trailing_zero32
 */