/* Template for the remote job exportation interface to GNU Make.
Copyright (C) 1988, 1989, 1992, 1993, 1996 Free Software Foundation, Inc.
This file is part of GNU Make.

GNU Make is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Make is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Make; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <pvm3.h>
#include <sys/wait.h>
#include <sys/resource.h>

#include "make.h"
#include "filedef.h"
#include "commands.h"
#include "debug.h"
#include "variable.h"
#include "pvm_message.h"
#include "autodepend.h"


char *remote_description = "PVMGmake-0.9 rpt PVM-" PVM_VER;

static int makelevel0=0; 
static int notify_max;
static int priority=0;

static int pvm_slots = 0;
static int mytid = -1;
static int nhosts = 0;
static int narchs = 0;

static struct pvmhostinfo *hostlist = 0;

typedef struct {
  struct pvmhostinfo *ref;
  int tid;
  int load;
  int loads;
  double avg[3];
} host_avg;

typedef struct {
  int tid;
  int task_status;
  int is_mine;
  int is_makefile;
  host_avg *host;
}Job;

static host_avg *avg_table = NULL;
static Job *jobs = NULL;
static int job_len=0;
static int used_slots=0;
static double avg = 0;

static int sigrequest = 0;
static char program_slave[256];
static char program_avg[256];

static char s_makelevel0 [256];
static char s_make_jobs [256];
     

/*
 Pending jobs management 
 */

typedef struct pending{
  struct pending *next;
  int tid;
  int flag;
  char where[256];
}Pending;

static Pending *pending_i,*pending_o,*pending_f;

int can_spawn() {
 return ( (used_slots < pvm_slots) && (
	      	((max_load_average >= 0 ) && (avg <= max_load_average)) || 
		(used_slots == 0) || 
		(max_load_average < 0)));
}
host_avg *
select_host (flag, where)
int	flag;
char *	where;
{
	host_avg  *best_host;
	int i,j,l;
	int starthost = 0;
      	DB(DB_JOBS,(_("[t%x] %s[%d]: Select Host %d %s \n"),mytid,program,makelevel,flag,where));

	/* Put load balancing code here */
	best_host = NULL;

	/* starting at 1 so that head node is not used 
	   but if there is only one node (does happen
	   in testing!) then give it a try */
	if (nhosts > 1)
		starthost = 1;
	for ( i = starthost; i<nhosts ; i++ ) {
	 
		if ( (flag & PvmTaskArch) && 
			strcmp(where, avg_table[i].ref->hi_arch) ) 
				continue;
		if ( (flag & PvmTaskHost) &&
			strcmp(where, avg_table[i].ref->hi_name) )
				continue;
		if ( best_host == NULL )
			best_host = &avg_table[i];

		if (max_load_average < 0) {
		  if ( avg_table[i].load < best_host->load )
		    best_host = &avg_table[i];
		} else {
		  if (( avg_table[i].avg[0] == best_host->avg[0] ) && ( avg_table[i].load < best_host->load ))
		    best_host = &avg_table[i];		  
		  
		  if ( avg_table[i].avg[0] < best_host->avg[0] )
		    best_host = &avg_table[i];		  
		}
			
	}
	
	if (best_host == NULL) return 0;
      	DB(DB_JOBS,(_("[t%x] %s[%d]: Host selected: %s load=%d Avg=%2.2f\n"),mytid,program,makelevel,best_host->ref->hi_name,best_host->load,best_host->avg[0]));
	return best_host;
}


void 
pending_push(tid,flag,where) 
    int tid;
    int flag;
    char *where;
{
  Pending *pending_n;
  
  if (pending_f == NULL) {
    int i;
    pending_f = (Pending*) malloc(sizeof(Pending)*pvm_slots);
    for (i = 0;i<pvm_slots-1;i++) pending_f[i].next = &pending_f[i+1];
    pending_f[i].next = NULL;
  }
  
  pending_n = pending_f;
  pending_f = pending_n->next;
  
  pending_n->tid = tid;
  pending_n->flag = flag;
  strcpy(pending_n->where,where);
  
  pending_n->next = NULL;
  if (pending_i == NULL) {
    pending_i = pending_o = pending_n;
  } else {
    pending_i->next = pending_n;
    pending_i = pending_n;
  }
  DB(DB_JOBS,(_("[t%x] %s[%d]: Pending jobs t%x\n"),mytid,program,makelevel,tid));
}

int 
pending_pop(tid,flag,where)
    int *tid;
    int *flag;
    char *where;
{
  Pending *pending_n;
  
  if (pending_o == NULL)
    return 0;
    
  pending_n = pending_o;
  
  pending_o = pending_n->next;
  if (pending_o == NULL) pending_i = NULL;
  
  pending_n->next = pending_f;
  pending_f = pending_n;
  
  *tid = pending_n->tid;
  *flag = pending_n->flag;
  strcpy(where,pending_n->where);
  
  return 1;
}
/*
  Check the return values 
*/

void check_pvm_return(ret)
    int ret;
{
  if (ret >=0) return;
  switch (ret) {
  case PvmSysErr: fatal(NILF,_("pvmd not responding (PvmSysErr)"));
  case PvmNoHost: fatal(NILF,_("Specified host is not in the virtual machine (PvmNoHost)"));
  case PvmNoFile: fatal(NILF,_("%s executable cannot be found (PvmNoFile)"),program_slave);
  default:fatal(NILF,_("PVM error (%d)"),ret);
  }
}

/*
 SIG management 
 */
void sighandler (sig)
    int sig;
{
  fprintf(stderr,_("[t%x] %s[%d]: *** PVM:Sig(%d) registered ***\n"),mytid,program,makelevel,sig);
  sigrequest = sig;
  signal(sig,sighandler);
}

void sigstop() {
  int i;
  int sig = sigrequest;
  
  fprintf(stderr,_("[t%x] %s[%d]: *** PVM:Handle Sig(%d) ***\n"),mytid,program,makelevel,sigrequest);
  sigrequest = -1; 
  
  /* Kill all known children */
  
  for (i=0;i < job_len;i++) { 
    DB(DB_JOBS,(_("[t%x] %s[%d]: Kill -%d t%x\n"),mytid,program,makelevel,sig,jobs[i].tid));
    pvm_sendsig(jobs[i].tid, sig);
  }
  
  /* Say Spwan Error for waiting Makefiles and kill then */
  
  if (makelevel == 0) {
    int tid,flag;
    char where[256];
    while (pending_pop(&tid,&flag,where) ) {
	pvm_sendsig(tid, sig);
        int slave_id=-1;
	pvm_initsend( PvmDataDefault );
        pvm_pkint(&slave_id, 1, 1);	
        pvm_send(tid , pvmMakeSlotAck );
    }
  }
  
  fatal_error_signal (sig);
}

/* Call once at startup even if no commands are run.  */
void
remote_setup ()
{
  int i,ret;
  char * ptr;

  sprintf(program_slave,"%s_pvm",program);  
  sprintf(program_avg,"%s_avg",program);  

  
  mytid = pvm_mytid( );
  if (mytid <= 0 ) return;

  if (!ISDB(DB_JOBS)) pvm_setopt(PvmAutoErr,0);

  /* Added statement below for symbian */
  pvm_setopt(PvmRoute, PvmRouteDirect);
  /*
            TDMPACKET     1       Enables packet tracing
            TDMMESSAGE    2       Enables message tracing
            TDMSELECT     4       Enables select and fd sets tracing
            TDMROUTE      8       Enables message route control tracing
            TDMSIZE       16      Enables read/write sizes tracing
            TDMSETSOCK    32      Enables setsockopt tracing
  */
  pvm_setopt(PvmDebugMask, 1 & 2 & 4 & 8 & 16 & 32 );
  
  check_pvm_return(mytid);
    
  DB(DB_JOBS,(_("[t%x] %s[%d]: My TID is t%x\n"),mytid,program,makelevel,mytid));
 
  pvm_config(&nhosts, &narchs, &hostlist);

  if (makelevel==0){
     struct variable *v;
     sprintf(s_makelevel0,"%d",mytid);
     v = define_variable ("MAKELEVEL0", 10, s_makelevel0, o_default, 0);
     v->export = v_export;
     makelevel0 = mytid;
    
  } else {
     int sid;
     ptr = getenv ("MAKELEVEL0");
     if (ptr) {
       makelevel0 = atoi(ptr);
     }
     
     ptr = getenv ("PVMMAKE_TID");
     if (ptr) {
       sid = atoi(ptr);
     }
     pvm_initsend( PvmDataDefault );
     pvm_pkint(&sid, 1, 1);	
     pvm_send(makelevel0,pvmMakeSlotIsMakefile);
  }
   
/*
 * calculate the number of slot
 */
  
  ptr = getenv ("MAKE_JOBS");
  if (ptr) {
    pvm_slots = atoi(ptr);
  } else {
     struct variable *v;
     if (job_slots) {
       pvm_slots = job_slots;
     }else {
       /*
        * calculate the number of slot
       */
       int min,sum;
       sum = min = hostlist[0].hi_speed;
       for (i = 1;i < nhosts  ;i++ ) {
         min = (min < hostlist[i].hi_speed)?min:hostlist[i].hi_speed;
	 sum += hostlist[i].hi_speed;
       }
       pvm_slots = sum / min * 2;
     }
     
     sprintf(s_make_jobs,"%d",pvm_slots);
     v = define_variable ("MAKE_JOBS", 9, s_make_jobs, o_default, 0);
     v->export = v_export;
     makelevel0 = mytid;    
  }
  
  /* Setup HOST_LIST */
  if (makelevel==0)
  {
    int arch_len = 0,host_len = 0;
    char *host_list;
    struct variable *v;
    int hid;
    char *pvm_arch;
    
    hid = pvm_tidtohost(mytid);

    for (i = 0;i < nhosts  ;i++ ) {
      arch_len += strlen(hostlist[i].hi_arch) + 1;
      host_len += strlen(hostlist[i].hi_name) + 1;
      
      if (hostlist[i].hi_tid == hid)
        pvm_arch = hostlist[i].hi_arch;
    }
    
    host_list = (char*) malloc(arch_len + host_len + 1);
    if (host_list == NULL)
      fatal(NILF,_("Can't set the host list\n"));
      
    host_list[0] = 0;
    
    for (i = 0;i < nhosts  ;i++ ) {
      strcat(host_list,hostlist[i].hi_arch);strcat(host_list,"/");
      strcat(host_list,hostlist[i].hi_name);strcat(host_list," ");
    }

    v = define_variable ("HOST_LIST", 9, host_list, o_default, 0);
    v->export = v_export;
    
    v = define_variable ("PVM_ARCH", 8, pvm_arch, o_default, 0);
    v->export = v_export;
    
    /* LoadAvg */
    avg_table = (host_avg *) calloc(nhosts,sizeof(host_avg));
    if (avg_table==NULL) fatal(NILF,_("Can't alloc the LoadAvg list\n"));

    for(i = 0; i < nhosts;i++) {
      char *arg[2]={"1",NULL};

      avg_table[i].tid = -1;
      avg_table[i].ref = &hostlist[i];

      if (max_load_average >= 0) {
      	ret = pvm_spawn( program_avg, arg, PvmTaskHost, hostlist[i].hi_name, 1, &avg_table[i].tid );  
      	check_pvm_return(ret); 
	avg_table[i].loads = 0;
	avg_table[i].avg[0] = max_load_average + 1.0;
      }
    }
  }
  used_slots = 0;
  job_slots = pvm_slots;
  
  notify_max = pvm_slots*2 + 1; /* N slot for N level2 make and N levelX jobs */
  jobs = malloc(sizeof(Job)*(notify_max+1));
  
  signal(SIGHUP,sighandler);
  signal(SIGINT,sighandler);
  signal(SIGQUIT,sighandler);
  signal(SIGTERM,sighandler);
  
  if (ISDB(DB_JOBS)) pvm_catchout(stdout);
  DB(DB_JOBS,(_("[t%x] %s[%d]: use %d jobs\n"),mytid,program,makelevel,pvm_slots));

  priority = nice(0);
  DB(DB_JOBS,(_("[t%x] %s[%d]: nice %d\n"),mytid,program,makelevel,priority));
}

/* Called before exit.  */

void remote_cleanup ()
{  
  int i;
  if (mytid < 0) return;
  
    if (avg_table)
    for(i = 0; i < nhosts;i++) {
      if (avg_table[i].tid != -1) {
	DB(DB_JOBS,(_("[t%x] %s[%d]: kill loadavg t%x\n"),mytid,program,makelevel,avg_table[i].tid));
        pvm_kill(avg_table[i].tid);
      }
    }  
  
  if (mytid > 0 ) {
    for (i=0;i < job_len;i++) {
      DB(DB_JOBS,(_("[t%x] %s[%d]: kill job t%x\n"),mytid,program,makelevel,jobs[i].tid));
      pvm_kill(jobs[i].tid);
      }
  }
  
  DB(DB_JOBS,(_("[t%x] %s[%d]: pvm_exit\n"),mytid,program,makelevel));
  pvm_catchout(0);
  pvm_exit( );

}

/* Return nonzero if the next job should be done remotely.  */

int start_remote_job_p (first_p)
     int first_p;
{
  return ( mytid > 0);  /* only if PVM launched ! */
}

/*
  Start a remote job for a sub Makefile
 */
 
void start_job_for_slave(tid,flag,where)
    int tid;
    int flag;
    char *where;
{
  int ret;
  int slave_id;
  char slave_pid[20];
  char *slave_arg_1[2]={slave_pid,NULL};
  char *slave_arg_2[3]={slave_pid,"-d",NULL};
  char **slave_arg = slave_arg_1;
  host_avg* host_sel;

  if (ISDB(DB_JOBS)) slave_arg = slave_arg_2;
  
  sprintf(slave_pid,"%d",tid);
  DB(DB_JOBS,(_("[t%x] %s[%d]: Spawn (%s)\n"),mytid,program,makelevel,where));

  host_sel = select_host(flag, where);
  
  if (host_sel) 
    ret = pvm_spawn( program_slave, slave_arg, PvmTaskHost, host_sel->ref->hi_name, 1, &slave_id );
  else
    ret = pvm_spawn( program_slave, slave_arg, flag, where, 1, &slave_id );

  check_pvm_return(ret);

  DB(DB_JOBS,(_("[t%x] %s[%d]: Start the job t%x for t%x\n"),mytid,program,makelevel,slave_id,tid));
  
  if (slave_id < 0) { /* else pvmMakeSlotAck is sent by the slave itself */
    pvm_initsend( PvmDataDefault );
    pvm_pkint(&slave_id, 1, 1);	
    pvm_send(tid , pvmMakeSlotAck );
  } else {
    used_slots ++;
    host_sel->load ++;
    
    jobs[job_len].tid=slave_id;
    jobs[job_len].task_status=0;
    jobs[job_len].is_mine = 0;
    jobs[job_len].is_makefile = 0;
    jobs[job_len].host = host_sel;
    job_len ++;
  
    DB(DB_JOBS,(_("[t%x] %s[%d]: pvm_notify t%x\n"),mytid,program,makelevel,slave_id));
    pvm_notify( PvmTaskExit, pvmMakeChildDead, 1, &slave_id );
  }
}
/*
  Start a remote job
*/

int start_job()
{
  int slave_id;
  int ret;
  struct variable *target=NULL;
  int flag;
  char *where;
  char *slave_arg_1[2]={"0",NULL};
  char *slave_arg_2[3]={"0","-d",NULL};
  char **slave_arg = slave_arg_1;
  host_avg* host_sel;
  
  if (ISDB(DB_JOBS)) slave_arg = slave_arg_2;
  
  target = lookup_variable ("TARGET_HOST", 11);
  if (target) {
    flag = PvmTaskHost;
    where = target->value;
  } else {
    target = lookup_variable ("TARGET_ARCH", 11);
    if (target) {
      flag = PvmTaskArch;
      where = target->value;
    } else {
      flag = PvmTaskDefault;
      where = "*";
    }
  }

  if (makelevel != 0) {  /* if makelevel != 0, ask job to the jobserver */

    pvm_initsend( PvmDataDefault );
    pvm_pkint(&flag, 1, 1);	
    pvm_pkstr(where);
    pvm_send(makelevel0,pvmMakeSlotReq );
    
    /* Wait for the slave process */
        
    ret = pvm_recv( -1, pvmMakeSlotAck );
    
    check_pvm_return(ret);
  
    pvm_upkint (&slave_id, 1, 1);
  
    check_pvm_return(slave_id);

    jobs[job_len].tid=slave_id;
    jobs[job_len].task_status=0;
    jobs[job_len].is_mine = 1;
    jobs[job_len].is_makefile = 0;
    jobs[job_len].host = NULL;
    job_len ++;
      
    DB(DB_JOBS,(_("[t%x] %s[%d]: pvm_notify t%x\n"),mytid,program,makelevel,slave_id));
    pvm_notify( PvmTaskExit, pvmMakeChildDead, 1, &slave_id );
    return slave_id;
  }
  
  
  DB(DB_JOBS,(_("[t%x] %s[%d]: Spawn (%s)\n"),mytid,program,makelevel,where));
  
  host_sel = select_host(flag, where);
  if (host_sel) 
    ret = pvm_spawn( program_slave, slave_arg, PvmTaskHost, host_sel->ref->hi_name, 1, &slave_id );
  else
    ret = pvm_spawn( program_slave, slave_arg, flag, where, 1, &slave_id );
  
  DB(DB_JOBS,(_("[t%x] %s[%d]: Start the job t%x for me\n"),mytid,program,makelevel,slave_id));
	
  check_pvm_return(ret);
  check_pvm_return(slave_id);
  
  jobs[job_len].tid=slave_id;
  jobs[job_len].task_status=0;
  jobs[job_len].is_mine = 1;
  jobs[job_len].is_makefile = 0;
  jobs[job_len].host = host_sel;
  job_len ++;

  DB(DB_JOBS,(_("[t%x] %s[%d]: pvm_notify t%x\n"),mytid,program,makelevel,slave_id));
  pvm_notify( PvmTaskExit, pvmMakeChildDead, 1, &slave_id );

  used_slots ++;
  host_sel->load ++;
  
  return slave_id;
}

/* Start a remote job running the command in ARGV,
   with environment from ENVP.  It gets standard input from STDIN_FD.  On
   failure, return nonzero.  On success, return zero, and set *USED_STDIN
   to nonzero if it will actually use STDIN_FD, zero if not, set *ID_PTR to
   a unique identification, and set *IS_REMOTE to zero if the job is local,
   nonzero if it is remote (meaning *ID_PTR is a process ID).  */

int start_remote_job (argv, envp, stdin_fd, is_remote, id_ptr, used_stdin)
     char **argv, **envp;
     int stdin_fd;
     int *is_remote;
     int *id_ptr;
     int *used_stdin;
{
  int slave_id;
  int ret;
  int i,j,k;
  int b;

  DB(DB_JOBS,(_("[t%x] %s[%d]: Wait for a job\n"),mytid,program,makelevel));

  slave_id = start_job();

  DB(DB_JOBS,(_("[t%x] %s[%d]: Got the job t%x\n"),mytid,program,makelevel,slave_id));

  if (slave_id > 0) {
    
    pvm_initsend( PvmDataDefault );
    
    /*
      push the $cwd
    */
    pvm_pkstr(starting_directory);

    /*
      push the priority
    */
    pvm_pkint(&priority, 1, 1);

    /*
      push the argv
    */
    for (j = 0;argv[j];j++);
    
    pvm_pkint(&j, 1, 1);
    
    for (i = 0; i < j ; i++){
 	int argv_size = strlen(argv[i]);
	pvm_pkint(&argv_size, 1, 1);
	pvm_pkstr(argv[i]);
      }

#ifndef PVM_EXPORT_ONLY
    /*
      push the envp
    */
    for (i = 0,j = 0;envp[i];i++)
      if (strncmp(envp[i],"PVM",3))
	j++;
    
    pvm_pkint(&j, 1, 1);
    
    for (i = 0,k = 0; k < j ; i++)
      if (strncmp(envp[i],"PVM",3)) {
	int env_size = strlen(envp[i]);
	pvm_pkint(&env_size, 1, 1);
	pvm_pkstr(envp[i]);
	k++;
      }
#endif
    
    pvm_send(slave_id , pvmMakeMng );

    *is_remote = 1;
    *id_ptr = slave_id;
    *used_stdin = 0;
    return 0;
  } 
}

/* Get the status of a dead remote child.  Block waiting for one to die
   if BLOCK is nonzero.  Set *EXIT_CODE_PTR to the exit status, *SIGNAL_PTR
   to the termination signal or zero if it exited normally, and *COREDUMP_PTR
   nonzero if it dumped core.  Return the ID of the child that died,
   0 if we would have to block and !BLOCK, or < 0 if there were none.  */

int
remote_status (exit_code_ptr, signal_ptr, coredump_ptr, block)
     int *exit_code_ptr, *signal_ptr, *coredump_ptr;
     int block;
{
  static char buffer[PVMGMAKE_IOBUFFERSIZE + 1];
  int bufid;
  int msgtag;
  int tid;
  int size;
  int i,j;


  do {
    if (sigrequest > 0) {
      sigstop();
    }
    
   if (block) {
     bufid = pvm_recv (-1, -1);
   } else {
     bufid = pvm_nrecv (-1, -1);
   }

    check_pvm_return(bufid);
    
    if (bufid > 0 ) {
      pvm_bufinfo( bufid, &size, &msgtag,&tid );

      switch (msgtag) {
      /* 
        Oputput 
       */
      case pvmMakeStdout:
	pvm_upkstr (buffer);
	write(1,buffer,strlen(buffer));
	break;
      case pvmMakeStderr:
	pvm_upkstr (buffer);
	write(2,buffer,strlen(buffer));
	break;
#ifdef  ENABLE_AUTODEPEND
      /*
        Autodepend
       */
      case pvmMakeDepend:
	pvm_upkstr (buffer);
	remote_catch_access(tid,buffer);
        break;
#endif
      /*
        Slot management
       */
      case pvmMakeSlotReq: /* A slave asks for a job */
        {
	  int ret;
	  int son;
	  int flag;
	  char where[256];
	  pvm_upkint (&flag, 1, 1);
	  pvm_upkstr (where);
	  
	  if ( can_spawn() ) {
	    start_job_for_slave(tid,flag,where);
	  } else {
	    pending_push(tid,flag,where);
	  }
	}
        break;
      case pvmMakeSlotIsMakefile:/* The slave is a MAKE ( a no productive process)*/
        {
	  int sid;	
	  pvm_upkint (&sid, 1, 1);
	  
	  DB(DB_JOBS,(_("[t%x] %s[%d]: t%x(t%x) is a %s\n"),mytid,program,makelevel,tid,sid,program));

	  for (i = 0;i < job_len;i++)
            if (jobs[i].tid == sid) {
	      
	      int nid,flag;
	      char where[256];
	    
	      used_slots --;  /* Do not monopolize a job ! */
      	      if (jobs[i].host) {
	        jobs[i].host->load --;
      	        DB(DB_JOBS,(_("[t%x] %s[%d]: Host released (isMakefile): %s load=%d\n"),mytid,program,makelevel,jobs[i].host->ref->hi_name,jobs[i].host->load));
	      }

              if (jobs[i].is_mine) { /* if is mine keep track */
	        jobs[i].is_makefile = 1;
	      } else {               /* if not forget it ! */
	      	jobs[i] = jobs[--job_len];
		/* some ressource managers are buggy 
      	      	  DB(DB_JOBS,(_("[t%x] %s[%d]: pvm_notify2 t%x\n"),mytid,program,makelevel,sid));
		  pvm_notify( PvmTaskExit|PvmNotifyCancel, pvmMakeChildDead, 1, &sid );
		*/

	      }
	      
	      if (can_spawn() && pending_pop(&nid,&flag,where)) { 
	        start_job_for_slave(nid,flag,where);
	      }
	      
	      break;
            }
        }
        break;
      case pvmMakeLoadAvg: /* A task stops */
        {
	  int nid,flag;
	  char where[256];
	  int load_was_too_high;
	  
	  load_was_too_high = (avg > max_load_average);
	  
          avg = max_load_average + 1.0;
      	  for (i = 0;i< nhosts;i++) {

	    if (tid == avg_table[i].tid) {
	      pvm_upkint(&avg_table[i].loads,1,1);
	      pvm_upkdouble(avg_table[i].avg,3,1);
	      DB(DB_JOBS,(_("[t%x] %s[%d]: LoadAvg %-20s %2.2f:%2.2f:%2.2f\n"),mytid,program,makelevel,
	      	      	      	      	      	      	      	      	       avg_table[i].ref->hi_name,
									       avg_table[i].avg[0],
									       avg_table[i].avg[1],
									       avg_table[i].avg[2]));
	    }
	  
	    if (avg_table[i].avg[0] < avg) avg = avg_table[i].avg[0];
	  }
	  DB(DB_JOBS,(_("[t%x] %s[%d]: Best LoadAvg %2.2f\n"),mytid,program,makelevel,avg));
	
	  if (load_was_too_high) {
	    while ( can_spawn() && pending_pop(&nid,&flag,where)) { 
	      start_job_for_slave(nid,flag,where);
	    }
	  }
      	}
	break;
      case pvmMakeEndTask: /* A task stops */
        {
	  int status;	
	  
	  pvm_upkint (&status, 1, 1);
	  DB(DB_JOBS,(_("[t%x] %s[%d]: Exit t%x status %d\n"),mytid,program,makelevel,tid,status));

	  for (i = 0;i < job_len;i++)
            if (jobs[i].tid == tid) {
	      jobs[i].task_status = status;
	    }
	  break;
        }
        break;
      case pvmMakeChildDead: /* The jobs dies */
        {
	int task_tid;
	int i;
	int status;
	int is_mine,is_makefile;
	pvm_upkint (&task_tid, 1, 1);
	
        DB(DB_JOBS,(_("[t%x] %s[%d]: TaskExit t%x\n"),mytid,program,makelevel,task_tid));

	for (i = 0; i< job_len ; i++) {
          if (jobs[i].tid == task_tid) {
	    
	    is_mine = jobs[i].is_mine;
	    status = jobs[i].task_status;
	    is_makefile = jobs[i].is_makefile;
	    
	    if (is_makefile == 0){
	      used_slots --;
      	      if (jobs[i].host) {
	        jobs[i].host->load --;
      	        DB(DB_JOBS,(_("[t%x] %s[%d]: Host released (ChildDead): %s load=%d\n"),mytid,program,makelevel,jobs[i].host->ref->hi_name,jobs[i].host->load));
	      }
	    }
      
            jobs[i] = jobs[--job_len];
	    
	    if (is_makefile == 0){
	      int nid,flag;
	      char where[256];
	      if (pending_pop(&nid,&flag,where)) {
	        start_job_for_slave(nid,flag,where);
	      }
	    } 	    
	    
	    if (is_mine) {
	      if (WIFEXITED(status)) {
	        *exit_code_ptr = WEXITSTATUS(status);
	        *coredump_ptr = 0;
	        *signal_ptr=0;
	      } else if (WIFSIGNALED(status)) {
	        *exit_code_ptr = WEXITSTATUS(status);
	        *signal_ptr = WTERMSIG(status);
	        *coredump_ptr = status & 0x80;
	      }
	      return (task_tid);
	    } else {
	      break;
	    }
	  }	
        }}
        break;
      default:
      	break;
      }
    }
 
  } while (block && (bufid > 0));
  return (0);
}

/* Block asynchronous notification of remote child death.
   If this notification is done by raising the child termination
   signal, do not block that signal.  */
void
block_remote_children ()
{
  return;
}

/* Restore asynchronous notification of remote child death.
   If this is done by raising the child termination signal,
   do not unblock that signal.  */
void
unblock_remote_children ()
{
  return;
}

/* Send signal SIG to child ID.  Return 0 if successful, -1 if not.  */
int
remote_kill (tid, sig)
     int tid;
     int sig;
{
  DB(DB_JOBS,(_("[t%x] %s[%d]: Kill -%d t%x\n"),mytid,program,makelevel,sig,tid));

  return (pvm_sendsig( tid, sig));
}
