//
//  pointer_cast_test.cpp - a test for boost/pointer_cast.hpp
//
//  Copyright (c) 2005 Ion Gaztaaga
//  Copyright (c) 2005 Peter Dimov
//
// Distributed under the Boost Software License, Version 1.0. (See
// accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//
/*
 *  Portions copyright (c) 2006-2007 Nokia Corporation.  All rights reserved.
*/

#include <boost/pointer_cast.hpp>

#include <boost/shared_ptr.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/get_pointer.hpp>

#include <boost/detail/lightweight_test.hpp>

#ifdef __SYMBIAN32__
#include "std_log_result.h"
#define LOG_FILENAME_LINE __FILE__, __LINE__
#endif

namespace
{

// Let's create these inheritance relationship:
//
//    base   base2
//      |     |
//      derived
//         |
//  derived_derived
//

class base
{
   public:
   virtual ~base(){}
   int filler [5];
};

class base2
{
public:

    virtual ~base2(){}
    int filler [5];
};

class derived
   : public base, public base2
{
    int filler [5];
};

class derived_derived
   : public derived
{
    int filler [5];
};

// And now some simple check functions

template <class BasePtr>
bool check_dynamic_pointer_cast(const BasePtr &ptr)
{
   //Check that dynamic_pointer_cast versus dynamic_cast
   return
   //Correct cast with dynamic_pointer_cast
   boost::get_pointer(boost::dynamic_pointer_cast<derived>(ptr)) ==
      //Correct cast with dynamic_cast
      dynamic_cast<derived*>(boost::get_pointer(ptr))
   &&
   //Incorrect cast with dynamic_pointer_cast
   boost::get_pointer(boost::dynamic_pointer_cast<derived_derived>(ptr)) ==
      //Incorrect cast with dynamic_cast
      dynamic_cast<derived_derived*>(boost::get_pointer(ptr));
}

template <class BasePtr>
bool check_static_pointer_cast(const BasePtr &ptr)
{
   return
   //Cast base -> derived -> base2 using static_pointer_cast
   boost::get_pointer(
            boost::static_pointer_cast<base2>(
               boost::static_pointer_cast<derived>(ptr))) ==
   //Now the same with static_cast
   static_cast<base2*>(static_cast<derived*>(boost::get_pointer(ptr)));
}

template <class BasePtr>
bool check_const_pointer_cast(const BasePtr &ptr)
{
   return
   //Unconst and const again using const_pointer_cast
   boost::get_pointer(
      boost::const_pointer_cast<const base>
         (boost::const_pointer_cast<base>(ptr))) ==
   //Now the same with const_cast
   const_cast<const base*>(const_cast<base*>(boost::get_pointer(ptr)));
}

}

int main()
{
    std_log(LOG_FILENAME_LINE,"[Test Case for pointer_cast_test]");
    {
        // Try casts with shared_ptr

        boost::shared_ptr<base> ptr(new derived);

        BOOST_TEST( check_dynamic_pointer_cast( ptr ) );
        BOOST_TEST( check_static_pointer_cast( ptr ) );
        BOOST_TEST( check_const_pointer_cast( ptr ) );
    }

    {
        // Try casts with raw pointer

        boost::scoped_ptr<base> ptr(new derived);

        BOOST_TEST( check_dynamic_pointer_cast( ptr.get() ) );
        BOOST_TEST( check_static_pointer_cast( ptr.get() ) );
        BOOST_TEST( check_const_pointer_cast( ptr.get() ) );
    }

#ifdef __SYMBIAN32__
	int failures = boost::report_errors();
	if(failures)
	{
		std_log(LOG_FILENAME_LINE,"Result : Failed");
		assert_failed = true;
	}
	else
	{
		std_log(LOG_FILENAME_LINE,"Result : Passed");
	}
	std_log(LOG_FILENAME_LINE,"[End Test Case ]");
#endif
	testResultXml("pointer_cast_test");
	close_log_file();
	return failures;
}
