/*
* Copyright (c) 2006-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Implementation of shared memory (POSIX Standard).
*
*/


// INCLUDE FILES
#include <errno.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <unistd.h>
#include "sysreent.h"
 

// -----------------------------------------------------------------------------
// Funcation name: shmget
// Description: Get shared memory identifier using the IPC key generated by ftok.
// Returns: None zero number (shared memory identifier): On success
//          -1              : On error.
// In case of error, errno value set to
//          EACCES - Operation permission is denied to the calling process, see IPC.
//          EINVAL - The value of semid is not a valid semaphore identifier, or the
//                   value of semnum is less than 0 or greater than or equal to
//                   sem_nsems, or the value of cmd is not a valid command.
//          EPERM -  The argument cmd is equal to IPC_RMID or IPC_SET and the effective
//                   user ID of the calling process is not equal to that of a process
//                   with appropriate privileges and it is not equal to the value of
//                   sem_perm.cuid or sem_perm.uid in the data structure associated
//                   with semid.
//          ERANGE - The argument cmd is equal to SETVAL or SETALL and the value to
//                   which semval is to be set is greater than the system-imposed
//                   maximum.
// -----------------------------------------------------------------------------
//

extern "C" {

EXPORT_C int shmget(key_t key, int size, int shmflg)
    {
        return _shmget_r(&errno, key, size, shmflg);
    }

// -----------------------------------------------------------------------------
// Funcation name: shmat
// Description: Attaches the shared memory segment associated with the shared
//              memory identifier specified by shmid to the address space of
//              the calling process.
// Returns: Segment's start address: On success
//          -1   : On error.
// In case of error, errno value set to
//          EACCES - Operation permission is denied to the calling process.
//          EINVAL - The value of shmid is not a valid shared memory identifier;
//                   the shmaddr is not a null pointer and the value of
//                   (shmaddr-((ptrdiff_t)shmaddr%SHMLBA)) is an illegal
//                   address for attaching shared memory; or the shmaddr is not
//                   a null pointer, (shmflg&SHM_RND) is 0 and the value of
//                   shmaddr is an illegal address for attaching shared memory.
//          EMFILE - The number of shared memory segments attached to the calling
//                   process would exceed the system-imposed limit.
//          ENOMEM - The available data space is not large enough to accommodate
//                   the shared memory segment.
// -----------------------------------------------------------------------------
//
EXPORT_C void* shmat(int shmid, const void *shmaddr, int shmflg)
    {
    return _shmat_r(&errno, shmid, shmaddr, shmflg);
    }

// -----------------------------------------------------------------------------
// Funcation name: shmdt
// Description: Detaches the shared memory segment located at the address
//              specified by shmaddr from the address space of the calling process.
// Returns: 0    : On success.
//          -1   : On error.
// In case of error, errno value set to
//          EINVAL - The value of shmaddr is not the data segment start
//                   address of a shared memory segment.
// -----------------------------------------------------------------------------
//
EXPORT_C int shmdt(const void *shmaddr)
    {
    return _shmdt_r(&errno, shmaddr);
    }

// -----------------------------------------------------------------------------
// Funcation name: shmctl
// Description: Provides a variety of shared memory control operations as
//              specified by cmd.
// Returns: 0  : On success
//          -1 : On error.
// In case of error, errno value set to
//          EACCES - The argument cmd is equal to IPC_STAT and the calling
//                   process does not have read permission.
//          EINVAL- The value of shmid is not a valid shared memory
//                   identifier, or the value of cmd is not a valid command.
//          EPERM - The argument cmd is equal to IPC_RMID or IPC_SET and the
//                  effective user ID of the calling process is not equal to
//                  that of a process with appropriate privileges and it is
//                  not equal to the value of shm_perm.cuid or shm_perm.uid
//                  in the data structure associated with shmid.
//          EOVERFLO - The cmd argument is IPC_STAT and the gid or uid value
//                     is too large to be stored in the structure pointed to
//                     by the buf argument
// -----------------------------------------------------------------------------
//
EXPORT_C int shmctl(int shmid, int cmd, struct shmid_ds *buf)
    {
    return _shmctl_r(&errno, shmid, cmd, buf);
    }

} // extern "C" {

//  End of File
