/*
* Copyright (c) 2007 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Header file for EAacPlusUtil.cpp
*
*/


#ifndef 	EAACPLUSUTIL_H
#define 	EAACPLUSUTIL_H

// INCLUDES
#include <e32base.h>

// CLASS DECLARATION
class EAacpAudioFrameInfo;

/**
*  This class implements utility functions to extract AAC frame
*  information.
*
*  @lib MmfEAacPlusUtil.lib
*  @since 3.2
*/
class CEAacPlusUtil : public CBase
	{
	public:
        // CONSTANTS
        enum TEAacPlusFormatType
	        {
		    EAacPlusFormatRaw=0,
		    EAacPlusFormatAdif=1,
		    EAacPlusFormatAdts=2
        	};
        /**
        *  EAAC+ frame info struct.
        *
        *  @lib MmfEAacPlusUtil.lib
        *  @since 3.2
        */
        class TEAacPlusFrameInfo
            {
        public:
	        TInt    	iOutFrameSize;			// Size of PCM Samples generated by decoder
            TInt    	iNoOfSamples;			// Number of PCM Samples generated by decoder per frame
            TInt    	iSamplingFrequency;		// Sampling freq of AAC Code decoder
            TInt		iNoOfChannels;			// No of output channel 1=Mono; 2=Stereo
        	TInt		iObjectType;			// MPEG Audio Object Type 2=AAC-LC; 4=AAC-LTP
        	TInt		iOutSamplingFrequency;  // Sampling Frequence for the output samples
        	TInt		iDownSampledMode;    	// 0=Normal Mode; 1=Down Sampled mode
        	TInt		iSbr;	                // 0=SBR not present; 1=SBR present
        	TInt		iPs;	                // 0=PS not present; 1=PS present
            };

        class TEAacPlusFrameLenConfig
            {
        public:
			TEAacPlusFormatType iFormatType;	// Container type.
            TInt    	iSamplingFrequency;		// Sampling freq of AAC core decoder
        	TInt		iObjectType;			// MPEG Audio Object Type 2=AAC-LC; 4=AAC-LTP
            };


    public:  // Constructors and destructor

        /**
        * Two-phased constructor.
        */
        IMPORT_C static CEAacPlusUtil* NewL();

        /**
        * Destructor.
        */
        IMPORT_C virtual ~CEAacPlusUtil();

    public: // New functions

		/**
        * Used to get the format type.
        * @since 3.2
        * @param	aBuf                 buffer of AAC data from start of aac content
        * @param	aBufLen              length of aBuf
        * @param	TEAacPlusFormatType  returned format type 0=raw; 1=ADIF; 2=ADTS
        * @return   status
        */
        IMPORT_C TInt GetAacFormatType(const TUint8* aBuf, TInt aBufLen, TEAacPlusFormatType& aFormatType);

		/**
        * Used to get the audio format information for ADIF and ADTS content.
        * Data should be provided beginning with start of aac content including header.
        * Enough data should follow the header to determine SBR and PS if needed.
        * Can be used with mpeg data where aHdrBuf will contain the AudioSpecificConfig. Implicit SBR and PS
        * will not be determined.
        * @since 3.2
        * @param	aHdrBuf buffer with AAC data beginning at start of aac content including ADIF or ADTS header
        * @param	aHdrLen length of aHdrBuf; header length is returned in this parameter
        * @param	aFrameInfo returns the format information
        * @return	status
        */
		IMPORT_C TInt GetAacFrameInfo (const TUint8* aHdrBuf, TInt& aHdrLen, TEAacPlusFrameInfo& aFrameInfo);
		
		/**
        * Used to get the audio format information for mpeg aac.
        * aHdrBuf will contain the AudioSpecificConfig. aDataBuf will contain data at beginning of raw audio frames.
        * Implicit SBR and PS can be determined.
        * @since 3.2
        * @param	aHdrBuf buffer with AudioSpecificConfig data.
        * @param	aHdrLen length of aHdrBuf
        * @param	aFrameInfo returns the format information
        * @param	aDataBuf buffer with raw audio frames
        * @param	aDataBufLen length of aDataBuf
        * @return	status
        */
		IMPORT_C TInt GetAacFrameInfo (const TUint8* aHdrBuf, TInt& aHdrLen, TEAacPlusFrameInfo& aFrameInfo,
									TUint8* aDataBuf, TInt aDataBufLen);
									
		/**
        * Used to get the size in bytes of the frame found at the start of aBuf.
        * @since 3.2
        * @param	aBuf buffer with AAC data frame
        * @param	aBufLen length of aBuf
        * @param	aFrameLen returns the frame size in bytes
        * @return   status
        */
		IMPORT_C TInt GetAacFrameLength(const TUint8* aBuf, TInt aBufLen, TInt& aFrameLen);

		/**
        * Used to configure the utility with information needed to enable the apis.
        * This api can be used if GetAacFrameInfo is not used to initialize the utility.
        * @since 3.2
        * @param	aFormatType format type
        * @param	aSamplingFrequency sampling frequency
        * @return   status
        */
		IMPORT_C TInt Config(TEAacPlusFrameLenConfig& aFrameLenConfig);

    private:

        /**
        * C++ default constructor.
        */
        CEAacPlusUtil();

        /**
        * By default Symbian 2nd phase constructor is private.
        */
        void ConstructL();

 	private:	// Data
		TEAacPlusFormatType iFormat;
		EAacpAudioFrameInfo* iGetFrameLenParam;
		TBool iInitialized;
	};

#endif      // EAACPLUSUTIL_H

// End of File
