/*
* Copyright (c) 2007 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0""
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:               CPtiKeyMapData class definitions.
*
*/














#ifndef _PTI_KEYMAP_DATA_H
#define _PTI_KEYMAP_DATA_H

#include <e32base.h>
#include "PtiDefs.h"

//
// TPtiKeyBinding
//
class TPtiKeyBinding
    {
    public:
        TUint16 iScanCode;
        TUint16 iCase;
        TUint16 iIndex;
        TUint16 iNumChars;
    };

//
// TPtiVowelSequence
//
class TVowelSequence
	{
	public:
		TPtiKey iFirst;
		TPtiKey iSecond;
		TUint16 iChars[2];  // for upper and lower case
	};

/**
* Interface class for PtiEngine keymapping data. Keymapping data is built into ecom dlls
* and accessesed through this interface. PtiEngine clients do not normally need to use
* this class directly.
*/
class MPtiKeyMapData
	{
	public:
		/**
		* Returns language code for keymap data.
		*
		* @since S60 V5.0
		* @return Symbian OS language code.
		*/
		virtual TInt LanguageCode() const = 0;

		/**
		* Returns a boolean value indicating whether language dll contains keymapping 
		* data for given keyboard type.
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		
		* @return ETrue if data contains multitapping definitions.
		*         EFalse otherwise.
		*/
		virtual TBool HasKeyData(TPtiKeyboardType aKeyboardType) const = 0;

		/**
		* Returns a boolean value indicating whether key mapping data contains
		* dead keys.
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @return ETrue if data contains dead keys.
		*         EFalse otherwise.
		*/
		virtual TBool HasDeadKeys(TPtiKeyboardType aKeyboardType) const = 0;

		/**
		* Returns boolean value indicating whether keymapping data contains
		* vowel sequences (such as those needed for Vietnamese qwerty input).
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @return ETrue if data contains vowel sequences.
		*         EFalse otherwise
		*/
		virtual TBool HasVowelSequences(TPtiKeyboardType aKeyboardType) const = 0;

		/**
		* Returns keymap data for given keyboard type, key and text case.
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @param aKey  A key to be requested.
		* @param aCase A text case for aKey
		* @return Pointer to keymap data. 
		*/
		virtual TPtrC DataForKey(TPtiKeyboardType aKeyboardType,
		                         TPtiKey aKey,
		                         TPtiTextCase aCase) const = 0;
		/**
		* Returns keymap data for given keyboard type, key and text case. This version takes
		* an internal keymap binding table index as a parameter. PtiEngine framework
		* doesn't call this method directly. Normally the real implementation
		* for this method is generated by keymap data builder into keymap plugin
		* file, so each language data plugin has own implementation of this method.
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @param aIndex A internal index into keymap buildings table.
		* @return Pointer to keymap data. 
		*/
		virtual TPtrC DataForKey(TPtiKeyboardType aKeyboardType, TInt aIndex) const = 0;
	
		/**
		* Returns a boolean value indicating whether given key / case combination 
		* is so called 'dead key'.
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @param aKey A key to be queried.
		* @param aCase A text case for aKey
		* @return ETrue if given key / case combination is a dead key.
		*         EFalse otherwise.
		*/
		virtual TBool IsDeadKey(TPtiKeyboardType aKeyboardType,
		                        TPtiKey aKey,
		                        TPtiTextCase aCase) const = 0;
		/**
		* Returns dead key data for given key / case combination.
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @param aKey   A key to be queried.
		* @param aCase  A text case for aKey.
		* @return Pointer to dead key data.
		*/
		virtual TPtrC DeadKeyDataForKey(TPtiKeyboardType aKeyboardType,
		                                TPtiKey aKey,
		                                TPtiTextCase aCase) const = 0;
		/**
		* Returns pointer to vowel sequence table (needed for Vietnamese qwerty input).
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @param aNumEntries This integer will be filled with number of entries in table.
		* @return Pointer to vowel sequence table.
		*         NULL if there isn't vowel sequence data available. 
		*/
        virtual const TVowelSequence* VowelSequenceTable(TPtiKeyboardType aKeyboardType,
                                                         TInt& aNumEntries) const = 0;		
		/**
		* Returns maximum number of physical keys used for given keyboard type. 
		*
		* @since S60 V5.0
		* @param aKeyboardType Keyboard type.
		* @return number of physical keys used for multitap mode.
		*/		
		virtual TInt NumberOfKeys(TPtiKeyboardType aKeyboardType) const = 0;
		
		/**
		* This methods provides direct access to keymap data table. In most cases
		* this method is not needed outside keymap data classes.
		* Use only if none of the convinience methods above is applicable.
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @param aDataSize This integer filled with number of TUint16 values in data table.
		* @return Pointer to multitap data table.
		*/
	 	virtual const TUint16* KeyData(TPtiKeyboardType aKeyboardType,
	 	                               TInt& aDataSize) const = 0;				 	
	 	/**
	 	* This method provides direct access to keybinding table. Normally
	 	* this isn't needed outside of keymapping classes. Use only if none of the
	 	* convinienece methods above is applicable
	 	* 
	 	* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		 	
	 	* @param aNumItems This integer will be filled with number of items in reurned table.
	 	* @return Pointer to multitapping key bindings table.
	 	*/
	    virtual const TPtiKeyBinding* KeyBindingTable(TPtiKeyboardType aKeyboardType,
	                                                  TInt& aNumItems) const = 0;	    	       				    	    
	    /**
	    * This methods fills an array of key bindings with those keys that produce numeric
	    * mode characters for given keyboard type. This is needed because not all languages naturally
	    * contain  required numeric mode characters. Text input framework needs to know which
	    * keys to use in that case. For example, user always needs to be able to produce	 
	    * characters 'p' and 'w' for phone number editor, but many languages (Arabic, Hebrew, etc)
	    * don't have those characters in their qwerty key mapping data. This method makes it possible 
	    * to define in key mapping data file which keys will be used if a number mode character is
	    * missing from normal qwerty mappings.
	    *
	    * @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		    
	    * @param aResult resulting key array will be stored here. Existing entries are cleared.
	    */
	    virtual void GetNumericModeKeysL(TPtiKeyboardType aKeyboardType,
	                                     RArray<TPtiNumericKeyBinding>& aResult) const = 0;
	    
	    /**
	    * This methods takes the given TPtiNumericKeyBinding struct and fills its iChar
	    * field if iKey and iShifted parameters of that struct match to any of the
	    * qwerty number mode keys for this language. 
	    *
	    * @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		    
	    * @param aWhichKey A structure containing search parameters and result field.
	    * @return ETrue If matching number mode key was found.
	    *         EFalse Otherwise.
	    */
	    virtual TBool IsNumberModeKey(TPtiKeyboardType aKeyboardType,
	                                  TPtiNumericKeyBinding& aWhichKey) const = 0;
	    
	    /**
	    * Returns pointer to numeric mode key table for given keyboard type.
	    * Number of items in array is stored to given integer variable.
	    *
	    * @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		    
	    * @param aNumEntries The number of items in return array will be stored to this
		*                    integer value.
		* @return Pointer to numeric mode keys table  
	    */
	    virtual const TPtiNumericKeyBinding* NumericModeKeysTable(TPtiKeyboardType aKeyboardType,
	                                                              TInt& aNumEntries) const = 0;
	    	    
	    /**
	    * This method returns a boolean value indicating whether this language can be used
	    * in "Latin Only" editors. If this method returns EFalse, then input framework will
	    * substitute it locally with English for "Latin Only" editors. Default implementation
	    * returns ETrue, so most languages don't need to define this in their key mapping data file.
	    *
	    * @since S60 V5.0        	    
	    * @return ETrue if this language is suitable to be used in "Latin Only" -mode.
	    *         EFalse otherwise.	    
	    */
	    virtual TBool SuitableForLatinOnlyMode() const = 0;
	    
	    /**
	    * Returns pointer to dead key data table.	    
	    *
	    * @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		    
	    * @param aNumRowsInArray This integer is filled with number of rows in array.
	    * @return Pointer to dead key data array,
	    */
	    virtual const TUint16* DeadKeyDataArray(TPtiKeyboardType aKeyboardType,
	                                            TInt& aNumRowsInArray) const = 0;
	                 
        /**
	    * Returns boolean value indicating whether given keyboard data	    	    
	    * contains fn-key mappings.
	    *
	    * @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		    	     
	    * @return A boolean value indicating whether aKeyboardType contains fn-key
	    *         mappings
	    */	                                            	                                            
        virtual TBool HasFnKeyBindings(TPtiKeyboardType aKeyboardType) const = 0;	                                            
	    	 	    
	    /**
	    * Reserved vtable entries.
	    */ 
	    virtual void Reserved_1() = 0;	    
	    virtual void Reserved_2() = 0;
        virtual void Reserved_3() = 0;	    
	    virtual void Reserved_4() = 0;	    
	};


/**
* Default implementation for MPtiKeyMapData interface class.
*/
class CPtiKeyMapData : public CBase, public MPtiKeyMapData
	{
	public:		
		/**
		* Returns index of TPtiKeyBinding item in key binding table for given key.
		* Key binding table contains index of key's data in keymap data array as well as
		* number of characters tied to that key. When the index is know, the actual key
		* map data for a key can then be accesses through key binding table.  
		*
		* @since S60 V5.0
        * @param  aKeyboardType  Keyboard type			
		* @param aKey A key to be queried.
		* @return Index of aKey in key binding table.		          
		*/			
		IMPORT_C TInt CaseBasedIndexInBindingTable(TPtiKeyboardType aKeyboardType,
		                                           TPtiKey aKey,
		                                           TPtiTextCase aCase) const;
			    
	    /**
	    * Returns a key constant and text case associated to given index in 
	    * key binding table. 
	    * 
	    * @since S60 V5.0
        * @param  aKeyboardType  Keyboard type		    
	    * @param aIndex  An index to be queried.
	    * @param aCase   This value will be filled with text case associated to given
	    *                index.	    
	    * @return A key associated to given index. 
	    *         EPtiKeyNone if not found.                 
	    */	    
	    IMPORT_C TPtiKey KeyForBindingTableIndex(TPtiKeyboardType aKeyboardType,
	                                             TInt aIndex,
	                                             TPtiTextCase& aCase) const;
	                                             	
		/**
		* Reurns list of language codes for all found keymap data implementations.
		*
		* @since S60 V5.0
		* @param aResult An array to be filled with language codes.
		*/		    
	    IMPORT_C static void ListLanguagesL(RArray<TInt>& aResult);

		IMPORT_C ~CPtiKeyMapData();
		
		// From MPtiKeyMapData	
        IMPORT_C TInt LanguageCode() const;				
	    IMPORT_C TBool HasKeyData(TPtiKeyboardType aKeyboardType) const;	
		IMPORT_C TBool HasDeadKeys(TPtiKeyboardType aKeyboardType) const;
		IMPORT_C TBool HasVowelSequences(TPtiKeyboardType aKeyboardType) const;
		IMPORT_C TPtrC DataForKey(TPtiKeyboardType aKeyboardType,
		                         TPtiKey aKey,
		                         TPtiTextCase aCase) const;	
		IMPORT_C TPtrC DataForKey(TPtiKeyboardType aKeyboardType, TInt aIndex) const;		
		IMPORT_C TBool IsDeadKey(TPtiKeyboardType aKeyboardType,
		                        TPtiKey aKey,
		                        TPtiTextCase aCase) const;		                        
		IMPORT_C TPtrC DeadKeyDataForKey(TPtiKeyboardType aKeyboardType,
		                                TPtiKey aKey,
		                                TPtiTextCase aCase) const;
        IMPORT_C const TVowelSequence* VowelSequenceTable(TPtiKeyboardType aKeyboardType,
                                                         TInt& aNumEntries) const;			
		IMPORT_C TInt NumberOfKeys(TPtiKeyboardType aKeyboardType) const;			
	 	IMPORT_C const TUint16* KeyData(TPtiKeyboardType aKeyboardType,
	 	                               TInt& aDataSize) const;				 		 
	    IMPORT_C const TPtiKeyBinding* KeyBindingTable(TPtiKeyboardType aKeyboardType,
	                                                  TInt& aNumItems) const;	    	       				    	    	  
	    IMPORT_C void GetNumericModeKeysL(TPtiKeyboardType aKeyboardType,
	                                      RArray<TPtiNumericKeyBinding>& aResult) const;	    	  
	    IMPORT_C TBool IsNumberModeKey(TPtiKeyboardType aKeyboardType,
	                                  TPtiNumericKeyBinding& aWhichKey) const;	    	  
	    IMPORT_C const TPtiNumericKeyBinding* NumericModeKeysTable(TPtiKeyboardType aKeyboardType,
	                                                               TInt& aNumEntries) const;	    	    	  
	    IMPORT_C TBool SuitableForLatinOnlyMode() const;	    
	    IMPORT_C const TUint16* DeadKeyDataArray(TPtiKeyboardType aKeyboardType,
	                                             TInt& aNumRowsInArray) const; 
        IMPORT_C TBool HasFnKeyBindings(TPtiKeyboardType aKeyboardType) const;	    	                                             		
	                                            	                                                 
	    IMPORT_C void Reserved_1();	    
	    IMPORT_C void Reserved_2();	  
        IMPORT_C void Reserved_3();	    
	    IMPORT_C void Reserved_4();	 	      	    								
         
	private:
		TInt iReserved1;			        
        TInt iReserved2;		
	};


/**
* Keymap data factory class.
*/
class CPtiKeyMapDataFactory : public CBase
    {
    public:               
    	/**
		* Creates a key map data instance for given implementation uid.
		*
		* @since S60 V5.0
		* @param aImplUid An implemenation uid for key map data factory to be created.
		* @return Pointer to key map data factory interface.
		*         NULL if not found.
		*/			
		IMPORT_C static CPtiKeyMapDataFactory* CreateImplementationL(const TUid aImplUid);
		
		/**
		* Fills given list with implementation uids of all found key map data
		* factory implementations.
		*
		* @since S60 V5.0
		* @param aResult An array to be filled with uids.
		*/			
		IMPORT_C static void ListImplementationsL(RArray<TInt>& aResult);
        	
        /**
        * Returns keymap data object for given language.
        *
        * @since S60 5.0
        * @param aLanguageCode Languace code for requested data.
        * @return Pointer to keymap data obejct.
        *         NULL if no data available for given language.
        */        	
        virtual MPtiKeyMapData* KeyMapDataForLanguageL(TInt aLanguageCode) = 0;
        
        /**
        * Lists all languages supported by this data factory.
        * 
        * @since S60 5.0
        * @param aResult List instance for storing results.
        */
        virtual void ListLanguagesL(RArray<TInt>& aResult) = 0;

    	IMPORT_C ~CPtiKeyMapDataFactory();

        IMPORT_C virtual void Reserved_1();	    
	    IMPORT_C virtual void Reserved_2();
	    
    public:	    
	    inline TInt ImplementationUid() const;	 
	    
	private:	  		
		inline void SetDestructorKeyId(TInt aUid);
		inline void SetImplementationUid(TInt aUid);
    
    private:
    	TInt iDTorId;
    	TInt iImplUid;			        
        TInt iReserved;	
    };


// ---------------------------------------------------------------------------
// CPtiKeyMapDataFactory::SetDestructorKeyId
// 
// ---------------------------------------------------------------------------
//	
inline void CPtiKeyMapDataFactory::SetDestructorKeyId(TInt aUid)
	{
	iDTorId = aUid;
	}
		
// ---------------------------------------------------------------------------
// CPtiKeyMapDataFactory::ImplementationUid
// 
// ---------------------------------------------------------------------------
//	
inline TInt CPtiKeyMapDataFactory::ImplementationUid() const
    {
    return iImplUid;
    }	

// ---------------------------------------------------------------------------
// CPtiKeyMapDataFactory::SetImplementationUid
// 
// ---------------------------------------------------------------------------
//	
inline void CPtiKeyMapDataFactory::SetImplementationUid(TInt aUid)
    {
    iImplUid = aUid;
    }

#endif // _PTI_KEYMAP_DATA_H

// End of file
