/*
* Copyright (c) 2005 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  
*
*/


#ifndef CMCERTPSOURCE_H
#define CMCERTPSOURCE_H

// INCLUDES
#include <e32std.h>
#include <mcemediasource.h>

// CONSTANTS
const TMceSourceType KMceRTPSource = 1;
const TInt KMceJitterBufferLength = 10;
const TInt KMceJitterBufferTreshold = 5;

//FORWARD DECLARATIONS
class TMceIds;

// CLASS DECLARATION

/**
* Class representing incoming RTP stream in MCE.
*
* CMceRtpSource can be used to receive media data from network.
* It can be configured with stream specific values to avoid
* jitter generated by network. It may also be configured to
* recognize separate media bursts instead of just receiving
* continuous flow of data. 
*
* Events from this class can be received through class MMceRtpObserver.
*
* Enable and Disable operations to it are considered to be local, 
* so they are not signalled to remote terminal.
*
* One CMceRTPSource may contain several RTP streams which are identified with
* synchronization source (SSRC) values. Disabling or enabling CMceRtpSource
* will affect to all SSRCs but streaming from specific SSRC can be also
* controlled with EnableSSRCL and Disable SSRCL methods. These are also
* local operations, thus not signalled to remote terminal.
* 
*  @lib mceclient.lib
*/
class CMceRtpSource: public CMceMediaSource
	{

	public: // Constructors and destructor

	    /**
	    * Two-phased constructor.
	    * @param aBufferLength, length in packets of jitter buffer
	    * @param aBufferTreshold, treshold in packets of jitter buffer
	    * @param aInactivityTimer, timer (ms) to detect network inactivity.
	    *		 If 0, no activity detection, otherwise source is set to support
	    *        talk bursts and report inactivity.
	    * @param aIdentity, identity of user placed in RTCP, ownership not transferred.
	    */
		IMPORT_C static CMceRtpSource* NewL(
									  TUint aBufferLength = KMceJitterBufferLength,
									  TUint aBufferTreshold = KMceJitterBufferTreshold,
									  TUint32 aInactivityTimer = 0,
									  HBufC8* aIdentity = 0 );

	    /**
	    * Two-phased constructor.
	    * @param aBufferLength, length in packets of jitter buffer
	    * @param aBufferTreshold, treshold in packets of jitter buffer
	    * @param aInactivityTimer, timer (ms) to detect network inactivity.
	    *		 If 0, no activity detection, otherwise source is set to support
	    *        talk bursts and report inactivity.
	    * @param aIdentity, identity of user placed in RTCP, ownership not transferred.
	    */
		IMPORT_C static CMceRtpSource* NewLC(
									  TUint aBufferLength = KMceJitterBufferLength,
									  TUint aBufferTreshold = KMceJitterBufferTreshold,
									  TUint32 aInactivityTimer = 0,
									  HBufC8* aIdentity = 0 );
									 
	    /**
	    * Destructor.
	    */
		IMPORT_C ~CMceRtpSource();

	public: // From base class

	    /**
	    * Enables streaming explicitly.
	    */
		IMPORT_C void EnableL(); 

	    /**
	    * Disables streaming explicitly.
	    */
		IMPORT_C void DisableL(); 

	public: // Functions

	    /**
	    * Enables RTP inactivity timer.
	    * @param aInactivityTimer, timer value (ms) that after inactivity
		*		 of media from network is reported to client.
	    */
		IMPORT_C void EnableInactivityTimerL( TUint32 aInactivityTimer ); 

	    /**
	    * Disables RTP inactivity timer.
	    */
		IMPORT_C void DisableInactivityTimerL(); 

	    /**
	    * Updates settings for incoming session.
	    * @pre CMceSession::State() == CMceSession::EIncoming 
	    * @param aBufferLength, length in packets of jitter buffer
	    * @param aBufferTreshold, treshold in packets of jitter buffer
	    * @param aInactivityTimer, timer (ms) to detect network inactivity.
	    *		 If 0, no activity detection, otherwise source is set to support
	    *        talk bursts and report inactivity.
	    * @param aIdentity, identity of user placed in RTCP, ownership not transferred.
	    */
		IMPORT_C void UpdateL ( TUint aBufferLength = KMceJitterBufferLength,
								TUint aBufferTreshold = KMceJitterBufferTreshold,
								TUint32 aInactivityTimer = 0,
								HBufC8* aIdentity = 0 );
		
		/**
	    * Sends RTCP receiver report to recipient.
	    */
		IMPORT_C void SendRRL(); 

        /**
        * Returns the last RTP stream source identifier 
        * used in network (SSRC). 
        * @pre CMceSession::State() == CMceSession::EEstablished 
        * @return last source identifier used in network (SSRC).
        */
        IMPORT_C TUint32 Ssrc() const;

        /**
        * Gets an array of RTP synchronization source (SSRC) identifiers. 
        * Array includes all SSRC values this RtpSource has received RTP from.
        * Changes in array are notified using MMceRtpObserver interface.
        * @return array of synchronization source identifiers
        */ 
        IMPORT_C const RArray<TUint>& Ssrcs() const;			

        /**
        * Enables streaming from specified synchronization source (SSRC).
        * @param aSsrc synchronization source to be enabled
        */			
        IMPORT_C void EnableSsrcL( TUint aSsrc );

        /**
        * Disables streaming from specified synchronization source (SSRC).
        * @param aSsrc synchronization source to be disabled
        */			
        IMPORT_C void DisableSsrcL( TUint aSsrc );

        /**
        * Determines whether specific SSRC is enabled or not.
        * @param aSsrc the SSRC identifier
        * @return ETrue if specified SSRC is enabled; otherwise EFalse
        */			
        IMPORT_C TBool IsSsrcEnabledL( TUint aSsrc );

	private:

	    /**
	    * C++ default constructor.
	    */
		CMceRtpSource();

	    /**
	    * second-phase constructor
	    * @param aBufferLength, length in packets of jitter buffer
	    * @param aBufferTreshold, treshold in packets of jitter buffer
	    * @param aInactivityTimer, timer (ms) to detect network inactivity.
	    *		 If 0, no activity detection, otherwise source is set to support
	    *        talk burts and report inactivity.
	    * @param aMediaBurstIndicator, indicator sound to be played before
	    *        every new burst of media, if source in media burst mode.
	    */
		void ConstructL( TUint aBufferLength,
                         TUint aBufferTreshold,
                         TUint32 aInactivityTimer,
                         HBufC8* iMediaBurstIndicator );


	public: // Stub data
	
         RArray<TUint> iSsrcs;
	
	     TUint iBufferLength;
         TUint iBufferTreshold;
         TUint32 iInactivityTimer;
         TUint32 iSilenceTimer;
         TUint iToneFrequency;
         TUint iToneLength;
         HBufC8* iMediaBurstIndicator; 
         
	};


#endif
