/*
 * Copyright (c) 2007 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of "Eclipse Public License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description:  CS Server Client Session class
 *
 */

#ifndef __C_CS_SESSION_H
#define __C_CS_SESSION_H

// FORWARD DECLARATIONS
class CCsServer;
class CCsPluginInterface;
class CCsClientConversation;
class CCsConversationEvent;

// CLASS DECLARATIONS

/**
 * Represents a session (version 2) for a client thread on 
 * the server side
 */
class CCsSession : public CSession2
    {
public:

    /**
     * Two phase construction
     * 
     * @param aServer, instance of CsServer
     */
    static CCsSession* NewL(CCsServer* aServer);

    /**
     * ServiceL
     * Handles the servicing of client requests
     * Implements CSession2 ServiceL
     * 
     * @param aMessage  RMessage reference
     */
    void ServiceL(const RMessage2& aMessage);

    /**
     * ServiceError
     * Handles the leaves generated by ServiceL function
     * Implements CSession2 ServiceError
     * 
     * @param aMessage  RMessage instance
     * @parea, aError the error occured during the service 
     */
    void ServiceError(
            const RMessage2& aMessage, 
            TInt aError);

    /**
     * HandleNewConversationListEventL
     * Handles the new conversation list event received from cache
     * asynchronously
     * 
     * @param aClientConversation the client conversation to be processed
     */
    void HandleNewConversationListEventL(
            CCsClientConversation* aClientConversation);

    /**
     * HandleDeleteConversationListEventL
     * Handles the delete conversation  list event received from cache
     * asynchronously
     * 
     * @param aClientConversation the client conversation to be processed
     */
    void HandleDeleteConversationListEventL(
            CCsClientConversation* aClientConversation);

    /**
     * HandleModifyConversationListEventL
     * Handles the modify conversation modify event received from cache
     * asynchronously
     * 
     * @param aClientConversation the client conversation to be processed
     */
    void HandleModifyConversationListEventL(
            CCsClientConversation* aClientConversation);      

    /**
     * HandleNewConversationEventL
     * Handles the new conversation event received from cache
     * asynchronously
     * 
     * @param aClientConversation the client conversation to be processed
     */
    void HandleNewConversationEventL(
            CCsClientConversation* aClientConversation);

    /**
     * HandleDeleteConversationEventL
     * Handles the delete conversation event received from cache
     * asynchronously
     * 
     * @param aClientConversation the client conversation to be processed
     */
    void HandleDeleteConversationEventL(
            CCsClientConversation* aClientConversation);

    /**
     * HandleModifyConversationEventL
     * Handles the modify conversation event received from cache
     * asynchronously
     * 
     * @param aClientConversation the client conversation to be processed
     */
    void HandleModifyConversationEventL(
            CCsClientConversation* aClientConversation);

    /**
     * HandleChangeEventL
     * Handle on-the-fly data changes event
     * 
     * @param aConversation , conversation change
     * @param aEvent,  event for the change
     */
    void HandleChangeEventL(CCsClientConversation* aConversation,
            TUint32 aEvent );
    
    /**
     * HandleRefreshConversationListL.
     * Notify client to refresh conversation list.
     */
    void HandleRefreshConversationListL();
    
    /**
     * HandleRefreshConversationList.
     * Notify client to refresh conversations.
     */
    void HandleRefreshConversationL();

   
private:

    /**
     * Second phase constructor
     */
    void ConstructL();

    /**
     * Constructor
     */
    CCsSession(CCsServer* aServer);

    /**
     *  Destructor
     */
    ~CCsSession();

    /**
     * DoServiceL
     * Serve the client requests
     * 
     * @param aMessage, RMessage instace
     */
    void DoServiceL(const RMessage2& aMessage);

    /**
     * GetConversationsL
     * Service method to get the 
     * conversation entry list for one conversation entry ID
     * 
     * @param aMessage RMessage instace
     */		
    void GetConversationsL(const RMessage2& aMessage);

    /**
     * GetConversationListL
     * Service method to get the 
     * list of recent(latest) conversation entry and 
     * list of dispalyname for all stored conversation entry ID
     * 
     * @param aMessage RMessage instace
     */     
    void GetConversationListL(const RMessage2& aMessage);

    /**
     * GetConversationUnreadListL
     * Service method to get the
     * list of recent(latest) unread conversation entry and
     * list of dispalyname for all stored conversation entry ID
     *
     * @param aMessage RMessage instace
     */
    void GetConversationUnreadListL(const RMessage2& aMessage);

    /**
     * DeleteConversationL
     * Service method to delete a conversation
     * 
     * @param aMessage RMessage instace
     */     
    void DeleteConversationL(const RMessage2& aMessage);

    /**
     * ShutdownServerL
     * Service method to shutdown the server
     * 
     * @param aMessage RMessage instace
     */
    void ShutdownServerL(const RMessage2& aMessage);

    /**
     * RequestChangeEventL
     * Service method to request add as observer 
     * for on cache change events
     * 
     * @param aMessage RMessage instace
     */
    void RequestChangeEventL(const RMessage2& aMessage);

    /**
     * RemoveChangeEventL
     * Service method to remove as observer 
     * for on cache change events
     * 
     * @param aMessage RMessage instace
     */
    void RemoveChangeEventL(const RMessage2& aMessage);

    /**
     * GetCachingStatusL
     * Service method to request conversation server
     * to get caching status
     * 
     * @param aMessage RMessage instace
     */
    void GetCachingStatusL(const RMessage2& aMessage);

    /**
     * GetTotalUnreadCountL
     * Gets total unread conversation entries.
     *
     * @param aMessage RMessage instace
     */
    void GetTotalUnreadCountL(const RMessage2& aMessage);

    /**
     * SetConversationListChangeObserverL
     * Service method to request conversation server
     * to set conversation list change observer flag
     * 
     * @param aMessage RMessage instace
     */
    void SetConversationListChangeObserverL(const RMessage2& aMessage);  

    /**
     * ResetConversationListChangeObserverL
     * Service method to request conversation server
     * to reset conversation list change observer flag
     * 
     * @param aMessage RMessage instace
     */
    void ResetConversationListChangeObserverL(const RMessage2& aMessage);  

    /**
     * SetConversationChangeObserverL
     * Service method to request conversation server
     * to set conversation change observer flag for given
     * client conversation
     * 
     * @param aMessage RMessage instace
     */
    void SetConversationChangeObserverL(const RMessage2& aMessage);  

    /**
     * ResetConversationChangeObserverL
     * Service method to request conversation server
     * to reset conversation change observer flag for given
     * client conversation 
     * 
     * @param aMessage RMessage instace
     */
    void ResetConversationChangeObserverL(const RMessage2& aMessage);  

    /**
     * SetCachingStatusObserverL
     * Service method to request conversation server
     * to set caching status observer flag
     * 
     * @param aMessage RMessage instace
     */
    void SetCachingStatusObserverL(const RMessage2& aMessage);  

    /**
     * ResetCachingStatusObserverL
     * Service method to request conversation server
     * to reset caching status observer flag
     *  
     * @param aMessage RMessage instance
     */
    void ResetCachingStatusObserverL(const RMessage2& aMessage);  

    /**
     * NotifyClient
     * Notify client for cache change event
     * 
     * @param aConversationEvent instance of the event to be notified
     */		
    void NotifyClient(CCsConversationEvent* aConversationEvent);

    /**
     * GetConversationIdL
     * Get the conversation id.
     * 
     * @param aMessage RMessage instance
     */
    void GetConversationIdL(const RMessage2& aMessage);
    
    /**
     * GetConversationIdfromAddressL
     * Get the conversation id from address
     * 
     * @param aMessage RMessage instance
     */
    void GetConversationIdfromAddressL(const RMessage2& aMessage);
	
	/**
     * GetConversationFromMessageIdL
     * Get the conversation from message-id
     * 
     * @param aMessage RMessage instance
     */
    void GetConversationFromMessageIdL(const RMessage2& aMessage);
    
    /**
     * MarkConversationReadL
     * Service method to mark a conversation as read.
     * 
     * @param aMessage RMessage instance
    */
    void MarkConversationReadL(const RMessage2& aMessage);
    
private:

    /**
     * iServer
     * Reference to server (not owned)
     */
    CCsServer*  iServer;

    /**
     * Buf for doing IPC
     * Own
     */
    HBufC8* des;

    /**
     * iEventList
     * Temp list of cache change event 
     * Own
     */
    RPointerArray<CCsConversationEvent>*  iEventList;

    /**
     * iMonitoredConversation
     * Conversation monitored for changes.
     */
    CCsClientConversation*  iMonitoredConversation;

    /**
     * iAsyncReqRMessage
     * Store RMeessage received from clinet
     * for cache change event notification
     *  (NotOwned)
     */
    RMessagePtr2  iAsyncReqRMessage;

    /**
     * iBufferOverflow
     * boolean to make if overflow has occured
     */
    TBool iBufferOverflow;

    /**
     * iGetConversationBufferOverflow
     * boolean to make if overflow has occured in API GetConversation
     */
    TBool iGetConversationBufferOverflow;

    /**
     * iNotifyHandling
     * iAsyncReqRMessage is valid or not for
     * notification handling
     * Owned
     */
    TBool iNotifyHandling;

    /**
     * iConversationListChangeObserver
     * It will true if observer
     * registered for conversation List change event
     * Owned
     */        
    TBool  iConversationListChangeObserver;

    /**
     * iConversationChangeObserver
     * It will true if observer
     * registered for conversation change observer
     * Owned
     */
    TBool  iConversationChangeObserver;

    /**
     * iCachingChangeObserver
     * It will true if observer
     * registered for caching status change observer
     * Owned
     */
    TBool  iCachingChangeObserver;

    /**
     * iReqCnt
     * Maintains the count of requests which are being served
     */
    TInt iReqCnt;
    };

#endif // __C_CS_SESSION_H

