/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/04/15 19:07:10 $
 * $Revision: 1.18.2.1 $
 */

/*
 *	Content:	Interface file to standard UNIX-style entry points ...
 *
 *	NB:			This file implements some UNIX low level support.  These functions
 *				are not guaranteed to be 100% conformant.
 *
 */


#include <stdlib.h>
#include <errno.h>
#include <file_io.h>
#include <unistd.h>
#include <fcntl.h>
#include <crtl.h>
#include <startup.win32.h>
#include <sysenv_api.h>
#include <unistd_api.h>
#include <extras_io.h>
#include <extras.win32.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <lmcons.h>

#ifdef _WINSIOUX                    /*- mm 980119 -*/
	#include <WinSIOUX.h>			/*- mm 990423 -*/
#endif /* _WINSIOUX  */             /*- mm 980119 -*/

/*
 *	int chdir(const char *path)
 *
 *		Changes the current working directory 
 */
int _MSL_CDECL chdir(const char * dirname) _MSL_CANT_THROW
{
    char path[_MAX_PATH];
    char drive[4];
    
    if (SetCurrentDirectory(dirname))
    {
        GetCurrentDirectory(_MAX_PATH, path);
        if (path[1] == ':')
        {
            drive[0] = '=';
            drive[1] = path[0];
            drive[2] = ':';
            drive[3] = 0;
            SetEnvironmentVariable(drive, path);
        }
        return 0;
    }
    else
        return -1;
}


char* _MSL_CDECL getcwd(char * path, int maxlen) _MSL_CANT_THROW
{
     return __msl_getcwd(path,maxlen);
}


/*
 *	int isatty(int fildes)
 *
 *		Determines is a filestream is going to the console window.
 */
int _MSL_CDECL isatty(int fildes) _MSL_CANT_THROW
{
    return GetFileType((void *)_get_osfhandle(fildes)) == FILE_TYPE_CHAR;
}
															
															
/*
 *	int close(int fildes)
 *	
 *		Closes a file stream.
 */
int _MSL_CDECL close(int fildes) _MSL_CANT_THROW												
{
	return __msl_close(fildes);
}

char * _MSL_CDECL cuserid(char * string) _MSL_CANT_THROW
{
	char			*name = getlogin();

	if (name != NULL && string != NULL)
		strcpy(string, name);
	return (name == NULL ? NULL : string == NULL ? name : string);
}

#pragma mark -

static char *make_cmdline(const char *prog, const char *const* args) _MSL_CANT_THROW
{
	int len = 0;
	int i;
	char *cmdline, *ptr;
	
	/* figure length of command line */
	for (i = 0; args[i]; i++)
	{
		/* actual program is prog, not argv[0] */
		const char *arg = i ? args[i] : prog;
		if (strchr(arg, ' '))
			len += 2;
		len += strlen(arg) + 1;
	}
	cmdline = malloc((len+1) * sizeof(char));	/* EJS 031105 */
	if (!cmdline) {
		errno = ENOMEM;
		return 0L;
	}
	
	/* concatenate arguments */
	ptr = cmdline;
	for (i = 0; args[i]; i++)
	{
		/* actual program is prog, not argv[0] */
		const char *arg = i ? args[i] : prog;
		int quoted = (strchr(arg, ' ') != 0L);
		if (quoted) *ptr++ = '"';
		len = strlen(arg);
		memcpy(ptr, arg, len);
		ptr += len;
		if (quoted) *ptr++ = '"';
		*ptr++ = ' ';
	}
	*ptr = 0;
	
	return cmdline;
}

/*	Execute process (core) */

static int	__spawn(int mode, const char *prog, const char *const* args, const char *const* envp) _MSL_CANT_THROW
{
	char *cmdline = make_cmdline(prog, args);
	PROCESS_INFORMATION piProcInfo;
	DWORD crFlags;
	DWORD exitcode;
	int ret;
	
	/* derive flags */
	switch (mode)
	{
	case _P_OVERLAY:
	case _P_NOWAIT:
	case _P_WAIT:
		crFlags = 0;
		break;
	case _P_DETACH:
		crFlags = DETACHED_PROCESS;
		break;
	default:
		errno = EINVAL;
		return -1;
	}
	
	/* launch child */
	ret = __msl_system_core(0L /*comspec*/, cmdline, envp, 
					crFlags, &piProcInfo);
	
	/* if error, return */
	if (ret)
		return -1;

	/* release lock on thread */
	CloseHandle(piProcInfo.hThread);
		
	/* do what is appropriate */
	switch (mode)
	{
		/* overlay: kill self */
	case _P_OVERLAY:
		/* release lock on process */
		CloseHandle(piProcInfo.hProcess);
		exit(0);
		break;
		
		/* no wait: return PID of launched child */
	case _P_NOWAIT:
	case _P_DETACH:
		return (int)piProcInfo.hProcess;
		
		/* wait: wait for child to complete, return its exit code */
	case _P_WAIT:
		if (WaitForSingleObject(piProcInfo.hProcess, INFINITE) != WAIT_FAILED)
			GetExitCodeProcess(piProcInfo.hProcess, &exitcode);
		else
			exitcode = STILL_ACTIVE;
		CloseHandle(piProcInfo.hProcess);
		return exitcode;
	}
	return -1;	/* notreached */
}

/*
 *	int exec(const char *path, ...)
 *
 *		Launches an application and then quits the current app.
 */
int _MSL_CDECL exec(const char *path, ...) _MSL_CANT_THROW
{
	/* in MacOS usage, the variable arguments are ignored */
	const char *argv[2];
	argv[0] = path;
	argv[1] = 0L;
	return __spawn(_P_OVERLAY, path, argv, _environ);
}

/*	Execute w/envp */
int	_MSL_CDECL spawnve(int mode, const char *prog, const char *const* args, const char *const* envp) _MSL_CANT_THROW
{
	return __spawn(mode, prog, args, envp);
}

/*	Execute */
int	_MSL_CDECL spawnv(int mode, const char *prog, const char *const* args) _MSL_CANT_THROW
{
	return __spawn(mode, prog, args, _environ);
}

/*	Execute w/envp */
int	_MSL_CDECL execve(const char *prog, const char *const* args, const char *const* envp) _MSL_CANT_THROW
{
	return __spawn(_P_OVERLAY, prog, args, envp);
}

/*	Execute */
int	_MSL_CDECL execv(const char *prog, const char *const* args) _MSL_CANT_THROW
{
	return __spawn(_P_OVERLAY, prog, args, _environ);
}

/*	find file with one of given extensions */
static int findprog(const char *prog, const char *ext, char *fullpath, size_t fullmax) _MSL_CANT_THROW
{
	const char *eptr, *pptr;
	int plen = strlen(prog);
	int noext;
	
	/* get filename portion of path and see if it has an extension */
	pptr = strrchr(prog, '\\');
	if (!pptr) pptr = strrchr(prog, '/');
	if (!pptr) pptr = prog;
	noext = strchr(pptr, '.') == 0L;
	
	for (eptr = ext; eptr[1]; eptr += strlen(eptr)+1) 
	{
		if (plen + strlen(eptr) + 1 < fullmax) 
		{
			memcpy(fullpath, prog, plen);
			strcpy(fullpath + plen, eptr);
			if (access(fullpath, X_OK) == 0)
			{
				return 1;
			}
		}
		
		/* only try first "extension" ("") if prog has one */
		if (!noext)
			break;
	}
	
	return 0;
}

/*	Find an executable in a semicolon-separated path */
static int findprogpath(const char *prog, const char *ext, const char *path, char *fullpath, size_t fullmax) _MSL_CANT_THROW
{
	const char *semi;
	int len, mx, flen = strlen(prog);
	char file[_MAX_PATH];
	
	while (*path)
	{
		semi = strchr(path, ';');
		if (!semi) semi = path + strlen(path);
		if ((len = semi - path) + 1 < _MAX_PATH)
		{
			memcpy(file, path, len * sizeof(char));
			file[len++] = '\\';
			mx = _MAX_PATH - len - 1;
			if (flen < mx)
			{
				memcpy(file + len, prog, (flen + 1) * sizeof(char));
				if (findprog(file, ext, fullpath, fullmax))
					return 1;
			}
		}
		path = *semi ? semi + 1 : semi;
	}
	return 0;
}

static int findexec(const char *prog, char *fullpath, size_t fullmax) _MSL_CANT_THROW
{
	int plen;

	/* search for file with given extensions */	
	if (findprog(prog, "\0.com\0.exe\0.bat\0.cmd\0", fullpath, fullmax))
		return 1;
	
	/* search PATH for prog without drive or directory separators */
	plen = strlen(prog);
	if (prog[1] != ':' && strchr(prog, '\\') == 0 && strchr(prog, '/') == 0)
	{
		const char *path;
		
		/* look for file with given extensions in PATH */
		path = getenv("PATH");
		if (path)
		{
			if (findprogpath(prog, "\0.com\0.exe\0.bat\0.cmd\0", path, fullpath, fullmax))
				return 1;
		}
	}
	
	return 0;
}

/*	Execute in path w/envp */
int	_MSL_CDECL spawnvpe(int mode, const char *prog, const char *const *args, const char *const* envp) _MSL_CANT_THROW
{
	char fullpath[_MAX_PATH];
	if (findexec(prog, fullpath, sizeof(fullpath)))
		return spawnve(mode, fullpath, args, envp);
	else {
		errno = ENOENT;
		return -1;
	}
}

/*	Execute in path */
int	_MSL_CDECL spawnvp(int mode, const char *prog, const char *const *args) _MSL_CANT_THROW
{
	return spawnvpe(mode, prog, args, _environ);
}

/*	Execute in path w/envp */
int	_MSL_CDECL execvpe(const char *prog, const char *const *args, const char *const* envp) _MSL_CANT_THROW
{
	return spawnvpe(_P_OVERLAY, prog, args, envp);
}

/*	Execute in path */
int	_MSL_CDECL execvp(const char *prog, const char *const *args) _MSL_CANT_THROW
{
	return spawnvpe(_P_OVERLAY, prog, args, _environ);
}

/*	Make argv[] array from va_list */
static char **make_args(va_list *ap) _MSL_CANT_THROW
{
	char *ptr;
	char **args = 0; 
	int maxargs = 0; int argc = 0;
	int last = 0;

	maxargs = 16;
	args = (char **)malloc(maxargs * sizeof(char *));
	if (!args) {
		errno = ENOMEM;
		return 0L;
	}
	
	while ((ptr = va_arg(*ap, char *)))
	{
		if (argc >= maxargs)
		{
			maxargs += 16;
			args = (char **)realloc(args, maxargs * sizeof(char *));
			if (!args) {
				errno = ENOMEM;
				return 0L;
			}
		}
		args[argc++] = ptr;
	}
	args[argc] = 0L;
	
	return args;
}

static void free_args(char **args) _MSL_CANT_THROW
{
	free(args);
}

/*	Spawn w/arglist */
int _MSL_CDECL __msl_vspawnl(int mode, const char *prog, va_list va) _MSL_CANT_THROW
{
	char **args;
	int ret;

	args = make_args(&va);
	ret = spawnv(mode, prog, args);
	free_args(args);
	return ret;
}

/*	Spawn w/arglist w/envp */
int	_MSL_CDECL __msl_vspawnle(int mode, const char *prog, va_list va) _MSL_CANT_THROW
{
	char **args;
	char **envp;
	int ret;

	args = make_args(&va);
	envp = va_arg(va, char **);
	ret = spawnve(mode, prog, args, envp);
	free_args(args);
	return ret;
}

/*	Spawn w/arglist using path */
int _MSL_CDECL __msl_vspawnlp(int mode, const char *prog, va_list va) _MSL_CANT_THROW
{
	char **args;
	int ret;

	args = make_args(&va);
	ret = spawnvp(mode, prog, args);
	free_args(args);
	return ret;
}

/*	Spawn w/arglist w/envp using path */
int	_MSL_CDECL __msl_vspawnlpe(int mode, const char *prog, va_list va) _MSL_CANT_THROW
{
	char **args;
	char **envp;
	int ret;

	args = make_args(&va);
	envp = va_arg(va, char **);
	ret = spawnvpe(mode, prog, args, envp);
	free_args(args);
	return ret;
}

/*	Spawn w/arglist */
int _MSL_CDECL spawnl(int mode, const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnl(mode, prog, va);
	va_end(va);
	return ret;
}

/*	Spawn w/arglist w/envp */
int	_MSL_CDECL spawnle(int mode, const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnle(mode, prog, va);
	va_end(va);
	return ret;
}

/*	Spawn w/arglist, search PATH */
int _MSL_CDECL spawnlp(int mode, const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnlp(mode, prog, va);
	va_end(va);
	return ret;
}

/*	Spawn w/arglist w/envp, search PATH */
int	_MSL_CDECL spawnlpe(int mode, const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnlpe(mode, prog, va);
	va_end(va);
	return ret;
}

/*	Execute w/arglist */
int _MSL_CDECL execl(const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnl(_P_OVERLAY, prog, va);
	va_end(va);
	return ret;
}

/*	Execute w/arglist w/envp */
int	_MSL_CDECL execle(const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnle(_P_OVERLAY, prog, va);
	va_end(va);
	return ret;
}

/*	Execute w/arglist, search PATH */
int	_MSL_CDECL execlp(const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnlp(_P_OVERLAY, prog, va);
	va_end(va);
	return ret;
}

/*	Execute w/arglist w/envp, search PATH */
int	_MSL_CDECL execlpe(const char *prog, ...) _MSL_CANT_THROW
{
	va_list va;
	int ret;

	va_start(va, prog);
	ret = __msl_vspawnlpe(_P_OVERLAY, prog, va);
	va_end(va);
	return ret;
}

/*	Wait for process to terminate */
int _MSL_CDECL cwait(int *termstat, int pid, int action) _MSL_CANT_THROW
{
	DWORD exitcode;

	/* action is ignored */
	if (action != _WAIT_CHILD && action != _WAIT_GRANDCHILD)
	{
		errno = EINVAL;
		return -1;
	}

	/* wait for process to complete, then retrieve the exit code */
	if (!WaitForSingleObject((HANDLE)pid, INFINITE)
	||	!GetExitCodeProcess((HANDLE)pid, &exitcode))
	{
		__set_errno(GetLastError());
		if (_doserrno == ERROR_INVALID_HANDLE)
			errno = ECHILD;	/* modify error when pid is invalid */
		return -1;
	}

	/* return exit code if desired */
	if (termstat)
		*termstat = (int)exitcode;
		
	return pid;
}

#pragma mark -

/*
 *	char *getlogin(void)
 *
 *		Returns the current user's login name (owner name from Sharing Setup CP).
 */
char * _MSL_CDECL getlogin(void) _MSL_CANT_THROW
{
	/* EJS 020405 */
	static char userid[UNLEN + 1], *unam = 0L;
	DWORD len = sizeof(userid);
	if (GetUserName(userid, &len))
		unam = userid;
	else if (!(unam = getenv("USERNAME")))
		unam = "root";
	return unam;
}

/*
 *	long lseek(int fildes, long offset,int whence)
 *	
 *		Seek in a file (fildes is the MacOS refnum).
 */
long _MSL_CDECL lseek(int fildes, long offset,int whence) _MSL_CANT_THROW
{
    return __msl_lseek(fildes,offset,whence);			/*- cc 010601 -*/
}


/*
 *	int read(int fildes, void *buf, size_t count)
 *	
 *		Read from a file (fildes is the file's MacOS refnum).
 *  Note: POSIX standard defines return value as of type ssize_t but says may be used instead    
 */
int _MSL_CDECL read(int fildes, void * buf, size_t count) _MSL_CANT_THROW					
{
 	return __msl_read(fildes,buf,count);
}


/*
 *	int rmdir(const char *path)
 *
 *		Removes a directory (must be empty).
 */
int _MSL_CDECL rmdir(const char *path) _MSL_CANT_THROW
{
	return RemoveDirectory((LPTSTR)path) ? 0 : 1;
}


/*
 *	unsigned int sleep(unsigned int seconds)
 *
 *		Pauses program execution for seconds seconds.
 */
unsigned int _MSL_CDECL sleep(unsigned int seconds) _MSL_CANT_THROW
{
	Sleep(seconds * 1000);
	return 0;
}

/*
 *	char *ttyname(int fildes)
 *
 *		Returns the name of the associated terminal or NULL if none.
 */
char * _MSL_CDECL ttyname(int fildes) _MSL_CANT_THROW
{
#if _WINSIOUX
	static char *__TTYDeviceName = "WinSIOUX";
#else
	static char *__TTYDeviceName = "console";
#endif
	
	if ((fildes >= 0) && (fildes <= 2))
		return __TTYDeviceName;
	else
		return NULL;
}

/*
 *	int unlink(const char *path)
 *	
 *		Unlink (i.e. delete) a file.
 */
int _MSL_CDECL unlink(const char * path) _MSL_CANT_THROW
{
    if (DeleteFile((LPCTSTR)path))
    {
        return 0;
    }
    else
    {
		/* get error code	 */
		__set_errno(GetLastError());
        return -1;
    }
}


/*
 *	int write(int fildes, const void *buf, size_t count)
 *	
 *		Write to a file (fildes is the file's MacOS refnum).
 *  Note: POSIX standard defines return value as of type ssize_t but says may be used instead    
 */
int _MSL_CDECL write(int fildes, const void *buf, size_t count)	 _MSL_CANT_THROW			
{
	    return __msl_write(fildes, buf, count);		
}


/* check accessibility of a file */
int _MSL_CDECL access(const char *path, int mode) _MSL_CANT_THROW
{
	DWORD Attributes;
	
	if (path == NULL || path == "")
	{
		errno = ENOENT;
		return -1;
	}
	if (strlen(path) > MAX_PATH)
	{
		errno = ENAMETOOLONG;
		return -1;
	}
	Attributes = GetFileAttributes((LPCSTR)path);
	if (Attributes == -1)
	{
		errno = ENOENT;
		return -1;
	}
	
	/*	Test all the flags */
	if (mode & W_OK)
	{	/* can we write? */
		if (Attributes & FILE_ATTRIBUTE_READONLY)
		{
			errno = EACCES;
			return -1;
		}
	}
	if (mode & R_OK)
	{	/* can we read? */
		/* cannot fail ... ? */
	}
	if (mode & X_OK)
	{	/* can we execute? */
		DWORD type;
		if (!GetBinaryType((LPCSTR)path, &type))
		{
			errno = EACCES;
			return -1;
		}
	}

	/* All tests passed, or mode is F_OK (and we already know it exists) */
	return 0;
}

int _MSL_CDECL __msl_setfilesize(int fildes, long size) _MSL_CANT_THROW
{
	DWORD orig;
	HANDLE hand;
	
	hand = _HandleTable[fildes]->handle;
	
	if (!hand)
	{
		errno = ENOENT;
		return -1;
	}
	
	if ((orig = SetFilePointer(hand, 0, NULL, FILE_CURRENT)) == INVALID_SET_FILE_POINTER)
	{
		__set_errno(GetLastError());
		return -1;
	}
	
	if (size < orig)
		orig = size;
	
	if (SetFilePointer(hand, size, NULL, FILE_BEGIN) == INVALID_SET_FILE_POINTER ||
		!SetEndOfFile(hand) ||
		SetFilePointer(hand, orig, NULL, FILE_BEGIN) == INVALID_SET_FILE_POINTER)
	{
		__set_errno(GetLastError());
		return -1;
	}
	
	return 0;
}

/* Change record:
 * hh  971206 Commented out _doserrno because it is already defined in extra.c
 * hh  971206 Added (char*) cast
 * hh  980122 Replaced <windows.h> with the following TWO includes because it is seriously
 *            broken.  The following 2 includes must be carefully ordered as shown, because
 *            they are broken too.
 * hh  980122 commented out unused arg 2 places
 * mf  980303 merged in support for Michael's winsioux
 * mm  980518 Changes to revert _lseek, _read, _write, _access, _close, _unlink, 
 *			  _rmdir, _chdir, _getcwd to lseek, read, write, access, close etc.
 *		      while accommodating legacy code.  Complement of changes in unistd.win32.h 
 * mf  980518 wince changes
 * mm  981230 Gave some body to getlogin() and exec() to prevent crash MW08465 
 * mm  990423 Made proper use of WinSIOUX.h instead of having builtin WinSIOUX prototypes
 * mm  991216 Reimplemented access() to work with directories as well as ordinary files 
 * cc  000209 changed windef.h to #define WIN32_LEAN_AND_MEAN & #include <windows.h>
 * mm  000222 Change to prevent possible access beyond the of read buffer IR9912-1054-1    
 * cc  000518 fixed #includes
 * mm  000607 Changed definitions of _read and _write to accord with POSIX Standard.
 * cc  010517 Renamed _chdir/chdir, _getcwd/getcwd, _isatty/isatty, _close/close, _cuserid/cuserid, _exec/exec
 *			  _getlogin/getlogin, _lseek/lseek, _read/read, _rmdir/rmdir, _sleep/sleep, _ttyname/ttyname
 *            _unlink/unlink, _write/write, _access/access, _write/write
 * cc  010601 Made lseek call __msl_lseek, write call __msl_write, close call __msl_close,read call __msl_read,
 * 			  open call __msl_open 
 * cc  010605 Made getcwd call __msl_getcwd & added #include file_io
 * cc  010115 Remove extern long _get_osfhandle common prototype is in extras.win32.h
 * mm  020211 Changed test of returned attributes to allow other attributes.
 * ejs 020405 Make getlogin() more reliable
 * ejs 020416 Release lock on spawned process thread handle as soon as possible
 * ejs 030623 Fix incorrect implementation of access()
 * JWW 031017 Added __msl_setfilesize to use native OS calls for changing a file size
 * ejs 031105 Fix buffer overrun in make_cmdline()
 */
 