/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/01/26 21:32:33 $
 * $Revision: 1.41 $
 */

/*************************************************************************
*	Project...:	Standard C99 math library/ ANSI/ISO C Standard Lib		 *
*   Author..... Matthew D. Fassiotto                                     *
*	Purpose...:	to provide fast/accurate double precision implementations
                of the standard C double functions as specified by math.h
                This is intended for any cpu that natively executes or
                emulates the  X87 extended precision instruction set.
                
    Notes: This is not completely compliant with every detail of the
           ratified C99 standard nor are all the functions implemented.
           We are missing erf,erfc, lgamma, gamma, fma, et. al.. 
           Unless we hear demand for these there is no plan to 
           implement the remaining functions.  
            
           This header gets included automatically by math.h.  It is not a
           standalone header meant to be included directly by any application.
           We do not support any source file including this header directly.
           This header depends on macros and other types defined in math.h
           which must be defined before this header is inlcuded.  Therefore
           you are guaranteed this will NOT compile when included directly.
            
           The functions below have previously shipped in coff object files
           as part of the CodeWarrior Professional product(e.g. math.obj).
           Most of these have been moved into this header to:
            
            1. Take advantage of the MWerks compiler's ability to efficiently
               inline assembly blocks, which will eliminate the function call
               and allow other performance optimizations such as better 
               register allocation and reduced memory spillage.
               
            2. Allow this part of the library to be compiled into various
               object formats(such as elf or coff).
               
            3. Allow these functions to take on various calling conventions
               on the fly.  For instance, if targeting a K6 3DNOW! chip the
               statement "return x;" will be compiled so that x is returned
               in the mmx register "MM0" where normally x would be popped
               from the fpu stack.  
               
            4. Nice and easy to debug!
                
           
           Special thanks to Fred Tydeman for finding several difficult
           bugs!!
			
*************************************************************************/


#ifndef _MSL_MATH_X87_H
#define _MSL_MATH_X87_H

#ifndef _MSL_CMATH
#error This header may only be included from <cmath>
#endif

#if _MSL_C99													/*- mm 030702 -*/
#if _MSL_MATH_ERRHANDLING & MATH_ERRNO
	#include <cerrno>
#endif /* MSL_MATH_ERRHANDLING & MATH_ERRNO */	
#endif /* _MSL_C99 -*/											/*- mm 030702 -*/

_MSL_BEGIN_NAMESPACE_STD

	typedef long double float_t;
	typedef long double double_t;	

_MSL_END_NAMESPACE_STD

/* private functions */

_MSL_BEGIN_EXTERN_C

	/* Get the relation between two numbers */
	_MSL_IMP_EXP_C short _MSL_MATH_CDECL __relation(double , double );
	_MSL_IMP_EXP_C unsigned int _MSL_MATH_CDECL __count_leading_zero(unsigned int x);

	#if _MSL_USE_INLINE
	
		#pragma only_std_keywords off 		/* allow asm */
		#pragma volatile_asm off			/* optimize inline assembly */

		#define __EQUAL 0x4000
		#define __GREATERTHAN 0
		#define __LESSTHAN 0x0100
		#define __UNORDERED 0x4500

		_MSL_INLINE short _MSL_MATH_CDECL __relation(double x, double y) _MSL_CANT_THROW
		{
			short 	ret;
			asm
			{
			fld       qword ptr x
			fcomp     qword ptr y
			fnstsw    ax
			and       ax,__UNORDERED
			mov		  ret, ax
			fclex     
			}
			return	ret;
		}

		_MSL_INLINE unsigned int _MSL_MATH_CDECL __count_leading_zero(unsigned int x) _MSL_CANT_THROW
		{
			unsigned int bits;
			if (x == 0)
				return 32U;
			asm bsr eax, x
			asm mov bits, eax
			return 31U - bits;
		
		}
		
		_MSL_INLINE	int _MSL_MATH_CDECL isgreater(double x, double y) _MSL_CANT_THROW { return (!__relation(x, y)); }
		_MSL_INLINE	int _MSL_MATH_CDECL isgreaterequal(double x, double y) _MSL_CANT_THROW { return (!((__relation(x, y)>>8) & 1)); }
		_MSL_INLINE	int _MSL_MATH_CDECL isless(double x, double y) _MSL_CANT_THROW { return (__relation(x, y) == __LESSTHAN); }
		_MSL_INLINE	int _MSL_MATH_CDECL islessequal(double x, double y) _MSL_CANT_THROW { return ((__relation(x, y)%__UNORDERED) != 0);}  /* pmk 020617 */
		_MSL_INLINE	int _MSL_MATH_CDECL islessgreater(double x, double y) _MSL_CANT_THROW { return (__relation(x, y) <= __LESSTHAN); }
		_MSL_INLINE	int _MSL_MATH_CDECL isunordered(double x, double y) _MSL_CANT_THROW { return (__relation(x, y) == __UNORDERED); }
		
		#pragma volatile_asm reset			/* reset inline assembly optimization */
		#pragma only_std_keywords reset		/* no more asm */

	#endif /* _MSL_USE_INLINE */
	
_MSL_END_EXTERN_C

/* only for Pro8 -- uses new iasm features */
#if __MWERKS__ >= 0x3000
	/*	bring in asm code */
	#include <math_x87_inlines.h>
#endif

_MSL_BEGIN_NAMESPACE_STD
_MSL_BEGIN_EXTERN_C

	#if _MSL_USE_INLINE
	#if _MSL_C99			
		/* all of these are mere casting stubs. */
		_MSL_INLINE float _MSL_MATH_CDECL
			acosf(float x) _MSL_CANT_THROW { return (float)(acos)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			acosl(long double x) _MSL_CANT_THROW { return (long double)(acos)((double)(x)); }			
		_MSL_INLINE float _MSL_MATH_CDECL
			acoshf(float x) _MSL_CANT_THROW { return (float)(acosh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			acoshl(long double x) _MSL_CANT_THROW { return (long double)(acosh)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			asinf(float x) _MSL_CANT_THROW { return (float)(asin)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			asinl(long double x) _MSL_CANT_THROW { return (long double)(asin)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			asinhf(float x) _MSL_CANT_THROW { return (float)(asinh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			asinhl(long double x) _MSL_CANT_THROW { return (long double)(asinh)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			atanf(float x) _MSL_CANT_THROW { return (float)(atan)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			atanl(long double x) _MSL_CANT_THROW { return (long double)(atan)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			atanhl(long double x) _MSL_CANT_THROW { return (long double)(atanh)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			atanhf(float x) _MSL_CANT_THROW { return (float)(atanh)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			atan2f(float y, float x) _MSL_CANT_THROW { return (float)(atan2)((double)(y), (double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			atan2l(long double y, long double x) _MSL_CANT_THROW { return (long double)(atan2)((double)(y), (double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			cbrtl(long double x) _MSL_CANT_THROW { return (long double)(cbrt)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			ceilf(float x) _MSL_CANT_THROW { return (float)(ceil)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			ceill(long double x) _MSL_CANT_THROW { return (long double)(ceil)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			copysignf(float x, float y) _MSL_CANT_THROW { return (float)(copysign)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			copysignl(long double x, long double y) _MSL_CANT_THROW { return (long double)(copysign)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			cosf(float x) _MSL_CANT_THROW { return (float)(cos)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			cosl(long double x) _MSL_CANT_THROW { return (long double)(cos)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			coshf(float x) _MSL_CANT_THROW { return (float)(cosh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			coshl(long double x) _MSL_CANT_THROW { return (long double)(cosh)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			expf(float x) _MSL_CANT_THROW { return (float)(exp)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			expl(long double x)  _MSL_CANT_THROW{ return (long double)(exp)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			expm1f(float x) _MSL_CANT_THROW { return (float)(expm1)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			expm1l(long double x) _MSL_CANT_THROW { return (long double)(expm1)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			fabsf(float x) _MSL_CANT_THROW { return (float)(fabs)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fabsl(long double x) _MSL_CANT_THROW { return (long double)(fabs)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			fdimf(float x, float y) _MSL_CANT_THROW { return (float)(fdim)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fdiml(long double x, long double y) _MSL_CANT_THROW { return (long double)(fdim)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			floorf(float x) _MSL_CANT_THROW { return (float)(floor)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			floorl(long double x) _MSL_CANT_THROW { return (long double)(floor)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			fmaxf(float x, float y) _MSL_CANT_THROW { return (float)(fmax)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fmaxl(long double x, long double y) _MSL_CANT_THROW { return (long double)(fmax)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			fminf(float x, float y) _MSL_CANT_THROW { return (float)(fmin)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fminl(long double x, long double y) _MSL_CANT_THROW { return (long double)(fmin)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			fmodf(float x, float y) _MSL_CANT_THROW { return (float)(fmod)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fmodl(long double x, long double y) _MSL_CANT_THROW { return (long double)(fmod)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			frexpf(float x, int* y) _MSL_CANT_THROW { return (float)(frexp)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			frexpl(long double x, int* y) _MSL_CANT_THROW { return (long double)(frexp)((double)(x), (y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			hypotf(float x, float y) _MSL_CANT_THROW { return (float)(hypot)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			hypotl(long double x, long double y) _MSL_CANT_THROW { return (long double)(hypot)((double)(x), (double)(y)); }
		_MSL_INLINE int _MSL_MATH_CDECL
			ilogbf(float x) _MSL_CANT_THROW { return (ilogb)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			ldexpf(float x, int y) _MSL_CANT_THROW { return (float)(ldexp)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			ldexpl(long double x, int y) _MSL_CANT_THROW { return (long double)(ldexp)((double)(x), (y)); }
		_MSL_INLINE long long _MSL_MATH_CDECL
			llrintf(float x) _MSL_CANT_THROW { return (llrint)((double)(x)); }
		_MSL_INLINE long long _MSL_MATH_CDECL
			llrintl(long double x) _MSL_CANT_THROW { return (llrint)((double)(x)); }
		_MSL_INLINE long long _MSL_MATH_CDECL
			llroundf(float x) _MSL_CANT_THROW { return (llround)((double)(x)); }
		_MSL_INLINE long long _MSL_MATH_CDECL
			llroundl(long double x) _MSL_CANT_THROW { return (llround)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			logf(float x) _MSL_CANT_THROW { return (float)(log)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			logl(long double x) _MSL_CANT_THROW { return (long double)(log)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			log1pf(float x) _MSL_CANT_THROW { return (float)(log1p)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			log1pl(long double x) _MSL_CANT_THROW { return (long double)(log1p)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			log10f(float x) _MSL_CANT_THROW { return (float)(log10)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			log10l(long double x) _MSL_CANT_THROW { return (long double)(log10)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			log2f(float x) _MSL_CANT_THROW { return (float)(log2)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			log2l(long double x) _MSL_CANT_THROW { return (long double)(log2)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			logbf(float x) _MSL_CANT_THROW { return (float)(logb)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			logbl(long double x) _MSL_CANT_THROW { return (long double)(logb)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lrintf(float x) _MSL_CANT_THROW { return (lrint)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lrintl(long double x) _MSL_CANT_THROW { return (lrint)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lroundf(float x) _MSL_CANT_THROW { return (lround)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lroundl(long double x) _MSL_CANT_THROW { return (lround)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			modfl(long double x, long double* iptr) _MSL_CANT_THROW {
		  double iptrd;
		  long double result = (long double)modf((double)x, &iptrd);
		  *iptr = (long double)iptrd;
		  return result;
		}
		_MSL_INLINE float _MSL_MATH_CDECL
			nearbyintf(float x) _MSL_CANT_THROW { return (float)(nearbyint)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			nearbyintl(long double x) _MSL_CANT_THROW { return (long double)(nearbyint)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			nextafterl(long double x, long double y) _MSL_CANT_THROW { return (long double)(nextafter)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			nexttowardl(long double x, long double y) _MSL_CANT_THROW { return (long double)(nexttoward)((double)(x), (y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			powf(float x, float y) _MSL_CANT_THROW { return (float)(pow)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			powl(long double x, long double y) _MSL_CANT_THROW { return (long double)(pow)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			remainderf(float x, float y) _MSL_CANT_THROW { return (float)(remainder)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			remainderl(long double x, long double y) _MSL_CANT_THROW { return (long double)(remainder)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			remquof(float x, float y, int* z) _MSL_CANT_THROW { return (float)(remquo)((double)(x), (double)(y), (z)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			remquol(long double x, long double y, int* z) _MSL_CANT_THROW { return (long double)(remquo)((double)(x), (double)(y), (z)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			rintf(float x) _MSL_CANT_THROW { return (float)(rint)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			rintl(long double x) _MSL_CANT_THROW { return (long double)(rint)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			roundf(float x) _MSL_CANT_THROW { return (float)(round)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			roundl(long double x) _MSL_CANT_THROW { return (long double)(round)((double)(x)); }
		
		/* on x86, scalbn/scalbln are identical to ldexp */
		_MSL_INLINE double _MSL_MATH_CDECL
			scalbn(double x, int y) _MSL_CANT_THROW { return ldexp(x,y); }
		_MSL_INLINE double _MSL_MATH_CDECL
			scalbln(double x, long int y) _MSL_CANT_THROW { return ldexp(x,(int)y); }			
		_MSL_INLINE float _MSL_MATH_CDECL
			scalblnf(float x, long int y) _MSL_CANT_THROW { return (float)(scalbln)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			scalblnl(long double x, long int y) _MSL_CANT_THROW { return (long double)(scalbln)((double)(x), (y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			scalbnf(float x, int y) _MSL_CANT_THROW { return (float)(scalbn)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			scalbnl(long double x, int y) _MSL_CANT_THROW { return (long double)(scalbn)((double)(x), (y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			sinf(float x) _MSL_CANT_THROW { return (float)(sin)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			sinhf(float x) _MSL_CANT_THROW { return (float)(sinh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			sinhl(long double x) _MSL_CANT_THROW { return (long double)(sinh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			sinl(long double x) _MSL_CANT_THROW { return (long double)(sin)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			sqrtf(float x) _MSL_CANT_THROW { return (float)(sqrt)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			sqrtl(long double x) _MSL_CANT_THROW { return (long double)(sqrt)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			tanf(float x) _MSL_CANT_THROW { return (float)(tan)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			tanhf(float x) _MSL_CANT_THROW { return (float)(tanh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			tanhl(long double x) _MSL_CANT_THROW { return (long double)(tanh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			tanl(long double x) _MSL_CANT_THROW { return (long double)(tan)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			truncf(float x) _MSL_CANT_THROW { return (float)(trunc)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			truncl(long double x) _MSL_CANT_THROW { return (long double)(trunc)((double)(x)); }

	#endif /*_MSL_C99*/

	#endif /* _MSL_USE_INLINE */


_MSL_END_EXTERN_C
_MSL_END_NAMESPACE_STD

#endif /* _MSL_MATH_X87_H */

/* Change record:
 * mf  040598 updated fmax,fmin,fdim to C9X to return a non-nan value when
 *		      exactly one argument is a nan.
 * mf  042098 fixed round and roundtol
 * mf  060898 performance enhancements, __declspec(naked)
 *            whenever possible(shrinks codesize by about 10% when combined
 *            with other minor fixes(elimination of local variables etc.)
 * mf  061098 fixed bug in acosh(was using fmul instead of fmul st,st(1)(i.e. was popping)
 * mf  071098 rewrote copysign
 * mf  071398 bug in asinh(wasn't restoring ebp)
 * mf  111398 log10 wasn't popping argument from fp stack when errno occured
 * mf  072199 moved most math_wrapper.c functions into this file. I left a few
 *            in math_wrapper.c because they are too big to _INLINE (like pow).
 *            fixed bug in fmin(IR9907-1375)(completely rewrote fmin/max/dim
 *            math_wrapper.c routines were sharing an unprotected/volitale static(TEMP).
 *            Each routine now has it's own local to make it threadsafe.
 * mf  061300 rounding mode for modf was wrong.  should be truncate, haven't a clue how this
 *            was changed.
 * mf  061200 added #pragma only_std_keywords off
 * mf  061300 added some optimizations such as qword ptr[_ret_val]
 * mf  061400 logb/remainder had a stack leak, added a pop
 * ejs 082200 don't include inline assembly in precompiled header
 * ejs 010320 add more OPTIMIZE directives
 * cc  010410 updated to new namespace macros
 * JWW 010618 Use cname headers exclusively to prevent namespace pollution in C++
 * ejs 010716 Enable inlined math functions in precompiled headers in 2.5
 * cc  011115 Part of _INLINE vs __inline fix
 * ejs 020111 relation() is defined in object for x87, thus must be __cdecl
 * ejs 020124 reorganized headers, changed static constants to inline constants
 * ejs 020130 Added float_t and double_t to namespace std
 * ejs 020222 Exclude intrinsic functions from inlining.
 * ejs 020314 Change __SET_ERRNO__ to check _MSL_MATH_ERRHANDLING
 * hh  020603 Added no throw spec to functions
 * pmk 020617 corrected islessequal() output, should only return 1 or 0
 * ejs 020807 Added nexttoward/nextafter
 * ejs 020809 Renamed relation to __relation, removed nonstandard isequal()
 * cc  030217 Added in the inline of cbrtl
 * mm  030529 Removed definition of modf(), now implemented in math.c
 * mm  030702 Added C99 wrappers
 */
