/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/01/26 21:32:22 $
 * $Revision: 1.30 $
 */

#ifndef _MSL_AMD_K63DNOW_MATH_H
#define _MSL_AMD_K63DNOW_MATH_H

#if !defined(_MSL_MATH_X87_H) && !defined(_MSL_MATH_K63D_H) && !defined(_MSL_MATH_SSE_H)
#error This header may only be included from <math_XXX.h>
#endif

_MSL_BEGIN_NAMESPACE_STD
_MSL_BEGIN_EXTERN_C

_MSL_BEGIN_EXTERN_C

_MSL_END_EXTERN_C

	#if _MSL_USE_INLINE

		#define __PIO2__ 1.57079632679489661923132169163975f
		#define __PI  3.1415926535897932384626433832795f
		#define __two_over_pi .636619772367581343075535053490057f
		#define __four_over_pi 1.27323954473516268615107010698011f

		#define __SQRT_FLT_EPSILON__ 3.4526698300e-04f
		#define __sqrt2_m1  .41421356237f

		#pragma gcc_extensions on	/* allow local arrays with initializers*/
		#pragma c9x on				/* allow // comments */

		_MSL_INLINE float _MSL_MATH_CDECL atan2f( float y, float x)
		{
			// need to figure out which quadrant we're in first
			// note |atanf(y/x)| < pi/2 and will return quadrant 1 if y and x
			// have the same sign and quadrant 4 if the signs differ.
			const _INT32 _sign_x=(_INT32)(__UHI(x)&0x80000000);
			const _INT32 _sign_y=(_INT32)(__UHI(y)&0x80000000);
			if(_sign_x == _sign_y)
			{
				if(_sign_x)  //we know both x, y<0 and angle is in quad 3
					return atanf(y/x) -__PI;
				if(x)
					return atanf(y/x);
				else
					return __PIO2__ ;  
			}

			// signs differ
			if( x < 0.0f)  // (x,y) in quadrant 2
			{
				return __PI + atanf(y/x);
			}

			// (x,y) in quadrant 4
			if(x)
				return atanf(y/x);

			__HI(y)=_sign_y+ 0x3FC90FDB;
			return y;
		}  

		_MSL_INLINE float _MSL_MATH_CDECL coshf( float arg)
		{
			return .5f*(expf(arg)+ expf(-arg));
		}

		_MSL_INLINE float _MSL_MATH_CDECL sinhf( float arg)
		{
			if(fabsf(arg) > 1.192092896e-07F)
		    	return .5f*(expf(arg)- expf(-arg));
		    else
				return arg ;    	
		}	

		_MSL_INLINE float _MSL_MATH_CDECL tanhf( float arg)
		{
			if(fabsf(arg) > 1.192092896e-07F)	
				return 1.0f - 2.0f/(expf(2.0f*arg)+1.0f);
			else
			    return arg ;
		}

#if _MSL_C99											/*- mm 030528 -*/
		_MSL_INLINE float _MSL_MATH_CDECL acosf( float x)
		{
			return  __PIO2__ - asinf(x);
		}
#endif /* _MSL_C99 */									/*- mm 030528 -*/

		_MSL_INLINE float _MSL_MATH_CDECL asinf( float x)
		{
			if(fabsf(x) < 1.0f)
				return atanf(x/sqrtf(1.0f - x*x));
			else if(x == 1.0f)
				return __PIO2__ ;
			else if(x == -1.0f)
				return -__PIO2__ ;

		#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO)
			errno = EDOM;
		#endif
			return NAN;
		}

		#if 0
		#pragma mark -
		#endif

		_MSL_INLINE float _MSL_MATH_CDECL ceilf(float x)
		{
			_INT32 i=(_INT32)x;   
			float y=x-(float)i; 
			if((!(*(_INT32*)&y)) || (__HI(x)&0x7f800000) >= 0x4B800000) 
				return x ;               // x is already an int
			else if(__HI(x)&0x80000000)
			   return (float)i;                   
			return (float)++i;
		}		
					
		_MSL_INLINE float _MSL_MATH_CDECL floorf(float x)
		{
			_INT32 i=(_INT32)x;               // signed int 
			float y=(float)i-x;       // since order of evaluation is NOT guaranteed
				                   // this is not guaranteed to work with all compilers for -0
				                   // I currently have no "cheap" work around to this.                          
			                   
			if((!(*(_INT32*)&y)) || (__HI(x)&0x7f800000) >= 0x4B800000) 
				return x ;               // x is already an int
			else if(__HI(x)&0x80000000) 
				return (float)--i;  

			// x < 0 -> int conversion of x above rounded toward zero(so decrement)                
			return (float)i;
		}

		/* 
		ldexp:			
		when underflow code is enabled this will return results identical to the exteneded precision 
		fscale instruction on any x86 fpu 100% of the time. 			
		*/

		_MSL_INLINE float _MSL_MATH_CDECL frexpf(float x, int* exp)
		{
			_INT32 tmp_int;
			switch( __HI(x)&0x7f800000 )
			{
			case 0x7f800000:
			case 0:
				*exp=0;  // here if zero,inf,or nan
				return x;
			}

			tmp_int=(_INT32)(0x3F000000 + (__UHI(x)&0x807fffff));
			*exp=((__HI(x)&0x7F800000)>>23)-126;
			return *((float*)&tmp_int);
		}
		 
		_MSL_INLINE float _MSL_MATH_CDECL log10f( float x)
		{
			switch( __HI(x)&0xff800000 )
			{
			default:
			{
				#define __log10_2  0.301029995664f  
				float   __log10_sqrt2 = 0.150514997831990597606869447362247f;  

				//poly 2302--misses double result by > 1 ulp but < 2 ulp .0034%
				const float _log10_poly[] = 
				{
					0.868588961f,
					0.289530878375f,
					0.173569242f,
					0.1307240643f
				};
				float y,zsq;
				_INT32 _exp,tmp;

				if(__HI(x)&0x80000000)  
				{
				#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO)
					errno = EDOM;
				#endif
					return NAN;
				}

				_exp = (__HI(x) >> 23) - 126; 
				if(__HI(x)&=0x007fffff)
					__HI(x)|= 0x3f000000;
				else
					return --_exp* __log10_2;

				if( __HI(x) < 0x3F3504F3) 	          
					x+=__sqrt2_m1*x;
				else
					__log10_sqrt2=0.0f;    

				x = 1.0f -(2.0f/(x + 1.0f));
				zsq = x*x;
				y=(((_log10_poly[3]*zsq + _log10_poly[2])*zsq + _log10_poly[1])*zsq + _log10_poly[0])*x;
				tmp= (_INT32)y;         //truncated toward zero
				y-=(float)tmp;
				return (_exp*__log10_2 + tmp) + (y- __log10_sqrt2);
			}

			case 0x7f800000:
			case 0xff800000:
				if(__HI(x)&0x007fffff) 
					return x;
				else 
				{
					if(__HI(x)&0x80000000) 
					{
				#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO)
						errno = EDOM;
				#endif
						return NAN;
					}
					else
						return INFINITY;
				}

			case 0:  // will fix for denormals later
			case 0x80000000:
				return -INFINITY ;

			}//end of switch
		}

		#pragma c9x reset
		#pragma gcc_extensions reset

	#endif	/*_MSL_USE_INLINE*/
	
_MSL_END_EXTERN_C
_MSL_END_NAMESPACE_STD

#endif	/*_MSL_AMD_K63DNOW_MATH_H*/

/* Change record:
 * ejs 030809  Set errno for out-of-range values of log10f, asinf
 */