/* $FreeBSD: src/sys/sys/sem.h,v 1.29 2004/11/17 13:12:06 rwatson Exp $ */
/*	$NetBSD: sem.h,v 1.5 1994/06/29 06:45:15 cgd Exp $	*/

/*
 * *  * Portions Copyright (c) 2006 Nokia Corporation and/or its subsidiary(-ies). All rights reserved.
 * SVID compatible sem.h file
 *
 * Author:  Daniel Boulet
 */

#ifndef _SYS_SEM_H_
#define _SYS_SEM_H_

#include <sys/ipc.h>


struct sem {
	u_short	semval;		/* semaphore value */
	pid_t	sempid;		/* process ID of last operation */
	u_short	semncnt;	/* number of processes waiting for semval
                           to become greater than current value */
	u_short	semzcnt;	/* number of processes waiting for semval
                           to become 0 */
    };

struct semid_ds {
	struct ipc_perm	sem_perm;	/* operation permission struct */
	struct sem	*sem_base;	/* pointer to first semaphore in set */
	unsigned short	sem_nsems;	/* number of sems in set */
	time_t		sem_otime;	/* last operation time */
	long		sem_pad1;	/* SVABI/386 says I need this here */
	time_t		sem_ctime;	/* last change time */
    					/* Times measured in secs since */
    					/* 00:00:00 GMT, Jan. 1, 1970 */
	long		sem_pad2;	/* SVABI/386 says I need this here */
	long		sem_pad3[4];	/* SVABI/386 says I need this here */
};

/*
 * semop's sops parameter structure
 */
struct sembuf {
	unsigned short	sem_num;	/* semaphore # */
	short		sem_op;		/* semaphore operation */
	short		sem_flg;	/* operation flags */
};
#define SEM_UNDO	010000

/*
 * semctl's arg parameter structure
 */
union semun {
	int		val;		/* value for SETVAL */
	struct		semid_ds *buf;	/* buffer for IPC_STAT & IPC_SET */
	unsigned short	*array;		/* array for GETALL & SETALL */
};

/*
 * commands for semctl
 */
#define GETNCNT	3	/* Return the value of semncnt {READ} */
#define GETPID	4	/* Return the value of sempid {READ} */
#define GETVAL	5	/* Return the value of semval {READ} */
#define GETALL	6	/* Return semvals into arg.array {READ} */
#define GETZCNT	7	/* Return the value of semzcnt {READ} */
#define SETVAL	8	/* Set the value of semval to arg.val {ALTER} */
#define SETALL	9	/* Set semvals from arg.array {ALTER} */
#define SEM_STAT 10	/* Like IPC_STAT but treats semid as sema-index */
#define SEM_INFO 11	/* Like IPC_INFO but treats semid as sema-index */

/*
 * Permissions
 */
#define SEM_A		IPC_W	/* alter permission */
#define SEM_R		IPC_R	/* read permission */

#ifdef _KERNEL

/*
 * semaphore info struct
 */
struct seminfo {
	int	semmap,		/* # of entries in semaphore map */
		semmni,		/* # of semaphore identifiers */
		semmns,		/* # of semaphores in system */
		semmnu,		/* # of undo structures in system */
		semmsl,		/* max # of semaphores per id */
		semopm,		/* max # of operations per semop call */
		semume,		/* max # of undo entries per process */
		semusz,		/* size in bytes of undo structure */
		semvmx,		/* semaphore maximum value */
		semaem;		/* adjust on exit max value */
};
extern struct seminfo	seminfo;

/*
 * Kernel wrapper for the user-level structure
 */
struct semid_kernel {
	struct	semid_ds u;
	struct	label *label;	/* MAC framework label */
};

/* internal "mode" bits */
#define	SEM_ALLOC	01000	/* semaphore is allocated */
#define	SEM_DEST	02000	/* semaphore will be destroyed on last detach */

/*
 * Process sem_undo vectors at proc exit.
 */
void	semexit(struct proc *p);
#endif /* _KERNEL */

#ifndef _KERNEL
#include <sys/cdefs.h>
#include <sys/_types.h>

#ifndef _SIZE_T_DECLARED
typedef __size_t        size_t;
#define _SIZE_T_DECLARED
#endif

#ifndef _PID_T_DECLARED
typedef __pid_t         pid_t;
#define _PID_T_DECLARED
#endif

// FUNCTION PROTOTYPES


// FORWARD DECLARATIONS


// CLASS/STRUCT/FUNCTION DECLARATION

__BEGIN_DECLS

/*
* Get semaphore identifier using the IPC key generated by ftok.
*/

IMPORT_C int semget(key_t key, int nsems, int semflg);


/*
* Perform atomically a user-defined array of semaphore operations on the set of 
* semaphores associated with the semaphore identifier specified by the argument semid.
*/

IMPORT_C int semop(int semid, struct sembuf *sops, unsigned nsops);


/*
* Provides a variety of semaphore control operations as specified by cmd.
*/
IMPORT_C int semctl(int semid, int semnum, int cmd, ...);


__END_DECLS
#endif // #ifndef _KERNEL

#endif // _SYS_SEM_H_

//  End of File
